<?php

namespace App\Http\Controllers\User;

use Illuminate\Support\Str;
use App\Models\User\Service;
use Illuminate\Http\Request;
use App\Models\User\Language;
use App\Models\User\ServiceContent;
use Mews\Purifier\Facades\Purifier;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Response;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Validator;

class ServiceController extends Controller
{
  public function services()
  {
    $languageId = Language::where('is_default', 1)->where('user_id', Auth::guard('web')->user()->id)->pluck('id')->first();

    $serviceContents = ServiceContent::with('service')
      ->where('language_id', '=', $languageId)
      ->orderBy('service_id', 'desc')
      ->get();

    return view('user.services.index', compact('serviceContents'));
  }

  public function createService()
  {
    // get all the languages from db
    $information['languages'] = Language::where('user_id', Auth::guard('web')->user()->id)->get();

    return view('user.services.create', $information);
  }

  public function storeService(Request $request)
  {

    $rules = [
      'service_icon' => 'required',
      'details_page_status' => 'required',
      'serial_number' => 'required'
    ];
    $defaulLang = Language::where([['user_id', Auth::guard('web')->user()->id], ['is_default', 1]])->first();
    $languages = Language::where('user_id', Auth::guard('web')->user()->id)->get();

    //default language validation
    $rules[$defaulLang->code . '_title'] = 'required|max:255';
    $rules[$defaulLang->code . '_summary'] = 'required';
    if ($request->details_page_status == 1) {
      $rules[$defaulLang->code . '_content'] = 'required|min:15';
    }

    foreach ($languages as $language) {
      if (
        $request->input($language->code . '_title') ||
        $request->input($language->code . '_summary') ||
        ($request->details_page_status == 1 &&  $request->input($language->code . '_content')) ||
        $request->input($language->code . '_meta_keywords') ||
        $request->input($language->code . '_meta_description')
      ) {
        $rules[$language->code . '_title'] = 'required';
        $rules[$language->code . '_summary'] = 'required';
        if ($request->details_page_status == 1) {
          $rules[$language->code . '_content'] = 'required|min:15';
        }
      }


      $messages[$language->code . '_title.required'] = __('The title field is required for') . ' ' . $language->name . ' ' . __('language') . '.';

      $messages[$language->code . '_title.max'] = __('The title field cannot contain more than 255 characters for') . ' ' . $language->name . ' ' . __('language') . '.';

      $messages[$language->code . '_summary.required'] = __('The summary field is required for') . ' ' . $language->name . ' ' . __('language') . '.';

      if ($request->details_page_status == 1) {
        $messages[$language->code . '_content.required'] = __('The content field is required for') . ' ' . $language->name . ' ' . __('language') . '.';
        $messages[$language->code . '_content.min'] = __('The content field atleast have 15 characters for') . ' ' . $language->name . ' ' . ('language') . '.';
      }
    }
    $validator = Validator::make($request->all(), $rules, $messages);
    if ($validator->fails()) {
      return Response::json([
        'errors' => $validator->getMessageBag()->toArray()
      ], 400);
    }

    $service = new Service();
    $service->service_icon = $request->service_icon;
    $service->user_id = Auth::guard('web')->user()->id;
    $service->details_page_status = $request->details_page_status;
    $service->serial_number = $request->serial_number;
    $service->save();

    foreach ($languages as $language) {

      if (
        $request->input($language->code . '_title') ||
        $request->input($language->code . '_summary') ||
        $request->input($language->code . '_content') ||
        $request->input($language->code . '_meta_keywords') ||
        $request->input($language->code . '_meta_description')
      ) {

        $serviceContent = new ServiceContent();
        $serviceContent->user_id = Auth::guard('web')->user()->id;
        $serviceContent->language_id = $language->id;
        $serviceContent->service_id = $service->id;
        $serviceContent->title = $request[$language->code . '_title'];
        $serviceContent->slug = Str::slug($request[$language->code . '_title']);
        $serviceContent->summary = $request[$language->code . '_summary'];
        $serviceContent->content = $request->details_page_status == 1 ? Purifier::clean($request[$language->code . '_content'], 'youtube') : '';
        $serviceContent->meta_keywords = $request[$language->code . '_meta_keywords'];
        $serviceContent->meta_description = $request[$language->code . '_meta_description'];
        $serviceContent->save();
      }
    }

    Session::flash('success', __('Created Successfully'));
    return 'success';
  }

  public function updateFeaturedService(Request $request)
  {
    $service = Service::findOrFail($request->serviceId);
    if ($request->is_featured == 1) {
      $service->update(['is_featured' => 1]);
    } else {
      $service->update(['is_featured' => 0]);
    }

    Session::flash('success', __('Updated Successfully'));
    return redirect()->back();
  }

  public function editService($id)
  {
    // get all the languages from db
    $information['languages'] = Language::where('user_id', Auth::guard('web')->user()->id)->get();
    $information['service'] = Service::findOrFail($id);
    return view('user.services.edit', $information);
  }

  public function updateService(Request $request, $id)
  {
    $rules = [
      'service_icon' => 'required',
      'details_page_status' => 'required',
      'serial_number' => 'required'
    ];
    $languages = Language::where('user_id', Auth::guard('web')->user()->id)->get();
    $defaulLang = Language::where([['user_id', Auth::guard('web')->user()->id], ['is_default', 1]])->first();

    //default language validation
    $rules[$defaulLang->code . '_title'] = 'required|max:255';
    $rules[$defaulLang->code . '_summary'] = 'required';
    if ($request->details_page_status == 1) {
      $rules[$defaulLang->code . '_content'] = 'required|min:15';
    }
    foreach ($languages as $language) {
      $hasExistingContent = ServiceContent::where('service_id', $id)
        ->where('language_id', $language->id)
        ->exists();
      if (
        $hasExistingContent ||
        $request->input($language->code . '_title') ||
        $request->input($language->code . '_summary') ||
        ($request->details_page_status == 1 &&  $request->input($language->code . '_content')) ||
        $request->input($language->code . '_meta_keywords') ||
        $request->input($language->code . '_meta_description')
      ) {
        $rules[$language->code . '_title'] = 'required';
        $rules[$language->code . '_summary'] = 'required';
        if ($request->details_page_status == 1) {
          $rules[$language->code . '_content'] = 'required|min:15';
        }
      }
      $messages[$language->code . '_title.required'] = __('The title field is required for') . ' ' . $language->name . ' ' . __('language');

      $messages[$language->code . '_title.max'] = __('The title field cannot contain more than 255 characters for') . ' ' . $language->name . ' ' . __('language');
      $messages[$language->code . '_summary.required'] = __('The summary field is required for') . ' ' . $language->name . ' ' . __('language');
      if ($request->details_page_status == 1) {
        $messages[$language->code . '_content.required'] = __('The content field is required for') . ' ' . $language->name . ' ' . __('language');
        $messages[$language->code . '_content.min'] = __('The content field atleast have 15 characters for') . ' ' . $language->name . ' ' . __('language');
      }
    }

    $validator = Validator::make($request->all(), $rules, $messages);
    if ($validator->fails()) {
      return Response::json([
        'errors' => $validator->getMessageBag()->toArray()
      ], 400);
    }
    $service = Service::findOrFail($id);
    $service->update([
      'service_icon' => $request->service_icon,
      'details_page_status' => $request->details_page_status,
      'serial_number' => $request->serial_number
    ]);

    foreach ($languages as $language) {
      $serviceContent = ServiceContent::where('service_id', $id)
        ->where('language_id', $language->id)
        ->first();
      $hasExistingContent = ServiceContent::where('service_id', $id)
        ->where('language_id', $language->id)
        ->exists();
      if (
        $hasExistingContent ||
        $request->input($language->code . '_title') ||
        $request->input($language->code . '_summary') ||
        ($request->details_page_status == 1 &&  $request->input($language->code . '_content')) ||
        $request->input($language->code . '_meta_keywords') ||
        $request->input($language->code . '_meta_description')
      ) {
        $content = [
          'language_id' => $language->id,
          'user_id' => Auth::guard('web')->user()->id,
          'service_id' => $id,
          'title' => $request[$language->code . '_title'],
          'slug' => Str::slug($request[$language->code . '_title']),
          'summary' => $request[$language->code . '_summary'],
          'content' => $request->details_page_status == 1 ? Purifier::clean($request[$language->code . '_content'], 'youtube') : '',
          'meta_keywords' => $request[$language->code . '_meta_keywords'],
          'meta_description' => $request[$language->code . '_meta_description']
        ];
        if (!empty($serviceContent)) {
          $serviceContent->update($content);
        } else {
          ServiceContent::create($content);
        }
      }
    }

    Session::flash('success', __('Updated Successfully'));
    return 'success';
  }

  public function deleteService(Request $request)
  {
    $service = Service::where('id', $request->service_id)->first();
    if ($service->serviceContent()->count() > 0) {
      $contents = $service->serviceContent()->get();
      foreach ($contents as $content) {
        $content->delete();
      }
    }

    $service->delete();
    Session::flash('success', __('Deleted Successfully'));
    return redirect()->back();
  }

  public function bulkDeleteService(Request $request)
  {
    $ids = $request->ids;
    foreach ($ids as $id) {
      $service = Service::findOrFail($id);
      if ($service->serviceContent()->count() > 0) {
        $contents = $service->serviceContent()->get();
        foreach ($contents as $content) {
          $content->delete();
        }
      }
      $service->delete();
    }
    Session::flash('success', __('Deleted Successfully'));
    return 'success';
  }
}
