<?php

namespace App\Http\Controllers\User;

use App\Constants\Constant;
use App\Enums\ThemeVersion;
use DateTime;
use Carbon\Carbon;
use App\Models\User\Room;
use App\Models\User\Coupon;
use Illuminate\Support\Str;
use Illuminate\Http\Request;
use App\Models\BasicExtended;
use App\Models\User\Language;
use App\Http\Helpers\MegaMailer;
use App\Models\User\RoomAmenity;
use App\Models\User\RoomBooking;
use App\Models\User\RoomContent;
use App\Models\User\MailTemplate;
use App\Models\User\RoomCategory;
use App\Traits\MiscellaneousTrait;
use Illuminate\Support\Facades\DB;
use App\Exports\RoomBookingsExport;
use App\Models\User\OfflineGateway;
use Illuminate\Support\Facades\URL;
use App\Http\Controllers\Controller;
use App\Http\Helpers\Uploader;
use App\Http\Requests\CouponRequest;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Mail;
use Maatwebsite\Excel\Facades\Excel;
use Illuminate\Support\Facades\Route;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Response;
use App\Models\Language as AdminLanguage;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\Validator;
use App\Http\Requests\AdminRoomBookingRequest;
use App\Models\User\BasicSetting;
use App\Models\User\PackageBooking;
use App\Models\User\PaymentGateway as OnlineGateway;
use App\Rules\ImageMimeTypeRule;
use Mews\Purifier\Facades\Purifier;
use Pdf;


class RoomController extends Controller
{
  use MiscellaneousTrait;

  public function settings()
  {
    $data = DB::table('user_basic_settings')->select('room_rating_status', 'room_guest_checkout_status')
      ->where('uniqid', 12345)
      ->first();

    return view('user.rooms.settings', ['data' => $data]);
  }

  public function updateSettings(Request $request)
  {
    $rules = [
      'room_rating_status' => 'required',
      'room_guest_checkout_status' => 'required'
    ];
    $validator = Validator::make($request->all(), $rules);
    if ($validator->fails()) {
      return Response::json([
        'errors' => $validator->getMessageBag()->toArray()
      ], 400);
    }
    DB::table('user_basic_settings')->update([
      'room_rating_status' => $request->room_rating_status,
      'room_guest_checkout_status' => $request->room_guest_checkout_status
    ]);

    Session::flash('success', __('Updated Successfully'));
    return 'success';
  }


  public function coupons()
  {
    // get the coupons from db
    $information['coupons'] = Coupon::orderByDesc('id')->where('user_id', Auth::guard('web')->user()->id)->get();
    // also, get the currency information from db
    $information['currencyInfo'] = MiscellaneousTrait::getCurrencyInfoUser();
    $language = Language::where('user_id', Auth::guard('web')->user()->id)->first();
    $rooms = Room::where('user_id', Auth::guard('web')->user()->id)->get();
    $rooms->map(function ($room) use ($language) {
      $room['title'] = $room->roomContent()->where('language_id', $language->id)->pluck('title')->first();
    });
    $information['rooms'] = $rooms;
    return view('user.rooms.coupons', $information);
  }

  public function storeCoupon(CouponRequest $request)
  {

    $startDate = Carbon::parse($request->start_date);
    $endDate = Carbon::parse($request->end_date);
    if ($request->filled('rooms')) {
      $rooms = $request->rooms;
    }
    Coupon::create($request->except('start_date', 'end_date', 'rooms') + [
      'start_date' => date_format($startDate, 'Y-m-d'),
      'end_date' => date_format($endDate, 'Y-m-d'),
      'rooms' => isset($rooms) ? json_encode($rooms) : null,
      'user_id' => Auth::guard('web')->user()->id
    ]);

    Session::flash('success', __('Created Successfully'));
    return 'success';
  }

  public function updateCoupon(CouponRequest $request)
  {
    $startDate = Carbon::parse($request->start_date);
    $endDate = Carbon::parse($request->end_date);
    if ($request->filled('rooms')) {
      $rooms = $request->rooms;
    }
    Coupon::find($request->id)->update($request->except('start_date', 'end_date', 'rooms') + [
      'start_date' => date_format($startDate, 'Y-m-d'),
      'end_date' => date_format($endDate, 'Y-m-d'),
      'rooms' => isset($rooms) ? json_encode($rooms) : null
    ]);

    Session::flash('success', __('Updated Successfully'));
    return 'success';
  }

  public function destroyCoupon($id)
  {
    Coupon::find($id)->delete();
    return redirect()->back()->with('success', __('Deleted Successfully'));
  }

  public function amenities(Request $request)
  {
    // first, get the language info from db
    $language = Language::where('code', $request->language)->where('user_id', Auth::guard('web')->user()->id)->firstOrFail();
    $information['language'] = $language;
    // then, get the room amenities of that language from db
    $information['amenities'] = RoomAmenity::where('language_id', $language->id)
      ->orderBy('id', 'desc')->where('user_id', Auth::guard('web')->user()->id)
      ->get();
    // also, get all the languages from db
    $information['langs'] = Language::where('user_id', Auth::guard('web')->user()->id)->get();

    return view('user.rooms.amenities', $information);
  }

  public function storeAmenity(Request $request)
  {
    $rules = [
      'serial_number' => 'required',
      'status' => 'required'
    ];

    $messages = [];
    $languages = Language::where('user_id', Auth::guard('web')->user()->id)->get();
    foreach ($languages as $language) {
      $rules[$language->code . '_name'] = 'required|max:255';
      $messages[$language->code . '_name.required'] = __('The name field is required for') . ' ' . $language->name . ' ' . __('language');
    }
    $validator = Validator::make($request->all(), $rules, $messages);
    if ($validator->fails()) {
      return Response::json([
        'errors' => $validator->errors()->toArray()
      ], 422);
    }
    $uid = uniqid();
    foreach ($languages as $language) {
      $roomAmenity = new RoomAmenity();
      $roomAmenity->language_id =  $language->id;
      $roomAmenity->user_id =  Auth::guard('web')->user()->id;
      $roomAmenity->name = $request[$language->code . '_name'];
      $roomAmenity->indx  = $uid;
      $roomAmenity->status =  $request->status;
      $roomAmenity->serial_number =  $request->serial_number;
      $roomAmenity->save();
    }

    Session::flash('success', __('Created Successfully'));
    return 'success';
  }

  public function editAmenity($id)
  {
    $userId = Auth::guard('web')->user()->id;
    $data['languages'] = Language::where('user_id', $userId)->get();
    $data['data'] = RoomAmenity::query()
      ->where('user_id', $userId)
      ->findorFail($id);
    return view('user.rooms.edit_amenity', $data);
  }

  public function updateAmenity(Request $request)
  {
    $rules = [
      'serial_number' => 'required'
    ];

    $languages = Language::where('user_id', Auth::guard('web')->user()->id)->get();
    $messages = [];
    foreach ($languages as $language) {
      $rules[$language->code . '_name'] = 'required|max:255';
      $messages[$language->code . '_name.required'] = __('The name field is required for') . '  ' . $language->name . '  ' . __('language');
    }
    $validator = Validator::make($request->all(), $rules, $messages);
    if ($validator->fails()) {
      return Response::json([
        'errors' => $validator->errors()->toArray()
      ], 422);
    }

    foreach ($languages as $language) {
      $roomAmenity = RoomAmenity::query()
        ->where('user_id', Auth::guard('web')->user()->id)
        ->where('indx', $request->amenity_indx)->where('language_id', $language->id)->first();
      $roomAmenity->name = $request[$language->code . '_name'];
      $roomAmenity->serial_number =  $request->serial_number;
      $roomAmenity->status =  $request->status;
      $roomAmenity->save();
    }
    Session::flash('success', __('Updated Successfully'));
    return 'success';
  }

  public function deleteAmenity(Request $request)
  {
    RoomAmenity::ownedByUser()->findOrFail($request->amenity_id)->delete();
    Session::flash('success', __('Deleted Successfully'));
    return redirect()->back();
  }

  public function bulkDeleteAmenity(Request $request)
  {
    $ids = $request->ids;
    foreach ($ids as $id) {
      RoomAmenity::ownedByUser()->findOrFail($id)->delete();
    }
    Session::flash('success', __('Deleted Successfully'));
    return 'success';
  }


  public function categories(Request $request)
  {
    // first, get the language info from db
    $language = Language::where('code', $request->language)->where('user_id', Auth::guard('web')->user()->id)->firstOrFail();
    $information['language'] = $language;
    // then, get the room categories of that language from db
    $information['roomCategories'] = RoomCategory::where('language_id', $language->id)->where('user_id', Auth::guard('web')->user()->id)
      ->orderBy('id', 'desc')
      ->paginate(10);
    // also, get all the languages from db
    $information['langs'] = Language::where('user_id', Auth::guard('web')->user()->id)->get();
    return view('user.rooms.categories', $information);
  }

  public function storeCategory(Request $request)
  {
    $settings = BasicSetting::select('theme_version')->where('user_id', Auth::guard('web')->user()->id)->first();

    $rules = [
      'status' => 'required',
      'serial_number' => 'required',
    ];
    if ($settings->theme_version == ThemeVersion::ThemeThree->value) {
      $rules['category_image'] = [
        'required',
        new ImageMimeTypeRule()
      ];
    }
    $languages = Language::where('user_id', Auth::guard('web')->user()->id)->get();
    $messages = [];
    foreach ($languages as $language) {
      $slug = slug_create($request[$language->code . '_name']);
      $rules[$language->code . '_name'] =
        [
          'required',
          'max:255',
          function ($attribute, $value, $fail) use ($slug, $language) {
            $bis = RoomCategory::where('language_id', $language->id)->where('user_id', Auth::guard('web')->user()->id)->get();
            foreach ($bis as $key => $bi) {
              if (strtolower($slug) == strtolower($bi->slug)) {
                $fail('The name field must be unique for ' . $language->name . ' language.');
              }
            }
          }
        ];
      $messages[$language->code . '_name.required'] = __('The name field is required for') . ' ' . $language->name . ' ' . __('language');
    }
    $validator = Validator::make($request->all(), $rules, $messages);
    if ($validator->fails()) {
      return Response::json([
        'errors' => $validator->getMessageBag()->toArray()
      ], 400);
    }
    // section image
    
    $category_image = null;
    if ($request->hasFile('category_image')) {
      $category_image = Uploader::upload_picture(
        public_path(Constant::WEBSITE_ROOM_IMAGE),
        $request->file('category_image'),
      );
    }
    
    $uid = uniqid();
    foreach ($languages as $language) {
      $RoomCategory = new RoomCategory();
      $RoomCategory->name = $request[$language->code . '_name'];
      $RoomCategory->slug = slug_create($request[$language->code . '_name']);
      $RoomCategory->language_id  = $language->id;
      $RoomCategory->user_id  = Auth::guard('web')->user()->id;
      $RoomCategory->indx  = $uid;
      $RoomCategory->status  = $request->status;
      $RoomCategory->image  = isset($category_image) ? $category_image : null;
      $RoomCategory->serial_number  = $request->serial_number;
      $RoomCategory->save();
    }

    Session::flash('success', __('Created Successfully'));
    return 'success';
  }

  public function editCategory($id)
  {
    $userId = Auth::guard('web')->user()->id;

    $data['languages'] = Language::where('user_id', $userId)->get();
    $data['data'] = RoomCategory::query()
      ->where('user_id', $userId)
      ->findorFail($id);
    return view('user.rooms.edit_category', $data);
  }

  public function updateCategory(Request $request)
  {
    $settings = BasicSetting::select('theme_version')->where('user_id', Auth::guard('web')->user()->id)->first();
    $rules = [
      'status' => 'required',
      'serial_number' => 'required'
    ];
    $messages = [];
    $languages = Language::where('user_id', Auth::guard('web')->user()->id)->get();
    foreach ($languages as $language) {
      $slug = slug_create($request[$language->code . '_name']);
      $indx = $request->category_indx;
      $rules[$language->code . '_name'] = [
        'required',
        'max:255',
        function ($attribute, $value, $fail) use ($slug, $indx, $language) {
          $bis = RoomCategory::where('indx', '<>', $indx)
            ->where('language_id', $language->id)
            ->where('user_id', Auth::guard('web')->user()->id)
            ->get();
          foreach ($bis as $key => $bi) {
            if (strtolower($slug) == strtolower($bi->slug)) {
              $fail(__('The title field must be unique for') . ' ' . $language->name . ' ' . __('language.'));
            }
          }
        }
      ];
      $messages[$language->code . '_name.required'] = __('The name field is required for') . ' ' . $language->name . ' ' . __('language');
    }

    if ($settings->theme_version == ThemeVersion::ThemeThree->value) {
      $rules['category_image'] = [
        'nullable', new ImageMimeTypeRule() ];
    }

    $validator = Validator::make($request->all(), $rules, $messages);
    if ($validator->fails()) {
      return Response::json([
        'errors' => $validator->getMessageBag()->toArray()
      ], 400);
    }

    $category_image = null;
    if ($request->hasFile('category_image')) {
      $category_image = Uploader::upload_picture(
        public_path(Constant::WEBSITE_ROOM_IMAGE),
        $request->file('category_image'),
      );
    }
    
    foreach ($languages as $language) {
      $RoomCategory = RoomCategory::query()
        ->where('user_id', Auth::guard('web')->user()->id)
        ->where('indx', $request->category_indx)->where('language_id', $language->id)->first();

      if ($settings->theme_version == ThemeVersion::ThemeThree->value && !empty($category_image)) {
        Uploader::remove(public_path(Constant::WEBSITE_ROOM_IMAGE), $RoomCategory->image);
      }

      
      $RoomCategory->name = $request[$language->code . '_name'];
      $RoomCategory->slug = slug_create($request[$language->code . '_name']);
      $RoomCategory->language_id  = $language->id;
      $RoomCategory->user_id  = Auth::guard('web')->user()->id;
      $RoomCategory->status  = $request->status;
      $RoomCategory->image  = !empty($category_image) ? $category_image :  $RoomCategory->image;
      $RoomCategory->serial_number  = $request->serial_number;
      $RoomCategory->update();
    }

    Session::flash('success', __('Updated Successfully'));
    return 'success';
  }

  public function deleteCategory(Request $request)
  {
    $roomCategory = RoomCategory::ownedByUser()->findOrFail($request->category_id);

    if ($roomCategory->roomContentList()->count() > 0) {
      Session::flash('warning', __('First delete all the rooms of this category!'));
      return redirect()->back();
    }

    $roomCategory->delete();

    Session::flash('success', __('Deleted Successfully'));
    return redirect()->back();
  }

  public function bulkDeleteCategory(Request $request)
  {
    $ids = $request->ids;
    foreach ($ids as $id) {
      $roomCategory = RoomCategory::ownedByUser()->findOrFail($id);
      if ($roomCategory->roomContentList()->count() > 0) {
        Session::flash('warning', __('First delete all the rooms of those category!'));
        return 'success';
      }
      $roomCategory->delete();
    }
    Session::flash('success', __('Deleted Successfully'));
    return 'success';
  }


  public function rooms(Request $request)
  {
    $languageId = Language::where('code', $request->language)->where('user_id', Auth::guard('web')->user()->id)->pluck('id')->first();
    $roomContents = RoomContent::with('room')
      ->where('language_id', '=', $languageId)->where('user_id', Auth::guard('web')->user()->id)
      ->orderBy('room_id', 'desc')
      ->get();

    $currencyInfo = MiscellaneousTrait::getCurrencyInfoUser();
    return view('user.rooms.rooms', compact('roomContents', 'currencyInfo', 'languageId'));
  }

  public function createRoom(Request $request)
  {
    // get all the languages from db
    $information['languages'] = Language::where('user_id', Auth::guard('web')->user()->id)->get();

    return view('user.rooms.create_room', $information);
  }

  public function sliderImgageUpload(Request $request)
  {

    $img = $request->file('slider_images');
    $allowedExts = array('jpg', 'png', 'jpeg');
    $rules = [
      'slider_images' => [
        function ($attribute, $value, $fail) use ($img, $allowedExts) {
          $ext = $img->getClientOriginalExtension();
          if (!in_array($ext, $allowedExts)) {
            return $fail("Only png, jpg, jpeg images are allowed");
          }
        }
      ]
    ];
    $validator = Validator::make($request->all(), $rules);
    if ($validator->fails()) {
      return response()->json(['error' => $validator->errors()], 422);
    }

    if ($request->hasFile('slider_images')) {
      $filename = Uploader::upload_picture(public_path(Constant::WEBSITE_ROOM_SLIDER_IMAGE), $request->file('slider_images'));
    }
    return response()->json(['status' => 'success', 'file_id' => $filename]);
  }
  public function removeImage(Request $request)
  {
    $img = $request['imageName'];

    try {
      Uploader::remove(public_path(Constant::WEBSITE_ROOM_SLIDER_IMAGE), $img);

      return Response::json(['success' => __('The image has been deleted.')], 200);
    } catch (\Exception $e) {
      return Response::json(['error' => __('Something went wrong!')], 400);
    }
  }

  public function detachImage(Request $request)
  {
    $id = $request['id'];
    $key = $request['key'];
    $room = Room::query()->find($id);
    $sliderImages = json_decode($room->slider_imgs, true);
    if (count($sliderImages) == 1) {
      return Response::json(['message' => __('Sorry, the last image cannot be delete.')], 400);
    } else {
      $image = $sliderImages[$key];
      Uploader::remove(public_path(Constant::WEBSITE_ROOM_SLIDER_IMAGE), $image);
      array_splice($sliderImages, $key, 1);
      $room->update([
        'slider_imgs' => json_encode($sliderImages)
      ]);
      return Response::json(['message' => __('Deleted Successfully')], 200);
    }
  }

  public function storeRoom(Request $request)
  {

    $defaulLang = Language::where([['user_id', Auth::guard('web')->user()->id], ['is_default', 1]])->first();
    $languages = Language::where('user_id', Auth::guard('web')->user()->id)->get();

    $rules = [
      'status' => 'required',
      'category' => 'required',
      'bed' => 'required',
      'bath' => 'required',
      'rent' => 'required',
      'thumbnail_image' => 'required',
      'slider_images' => 'required',
      'max_guests' => 'nullable|numeric',
      'quantity' => 'required|numeric',
      'amenities' => 'required',
      'latitude' => 'nullable | numeric| between:-90,90',
      'longitude' => 'nullable | numeric| between:-180,180',
    ];


    //default language validation
    $rules[$defaulLang->code . '_title'] = 'required|max:255';
    $rules[$defaulLang->code . '_summary'] = 'required';
    $rules[$defaulLang->code . '_description'] = 'required|min:15';


    foreach ($languages as $language) {
      if (
        $request->input($language->code . '_title') ||
        $request->input($language->code . '_summary') ||
        $request->input($language->code . '_description') ||
        $request->input($language->code . '_meta_keywords') ||
        $request->input($language->code . '_meta_description')
      ) {

        $rules[$language->code . '_title'] = 'required';
        $rules[$language->code . '_summary'] = 'required';
        $rules[$language->code . '_description'] = 'required';
      }

      $messages[$language->code . '_title.required'] = __('The title field is required for') . ' ' . $language->name . ' ' . __('language');
      $messages[$language->code . '_title.max'] = __('The title field cannot contain more than 255 characters for') . ' ' . $language->name . ' ' . __('language');
      $messages[$language->code . '_summary.required'] = __('The summary field is required for') . ' ' . $language->name . ' ' . __('language');
      $messages[$language->code . '_description.required'] = __('The description field is required for') . ' ' . $language->name . ' ' . __('language');
      $messages[$language->code . '_description.min'] = __('The description field atleast have 15 characters for') . ' ' . $language->name . ' ' . __('language');
    }
    $validator = Validator::make($request->all(), $rules, $messages);
    if ($validator->fails()) {
      return Response::json([
        'errors' => $validator->getMessageBag()->toArray()
      ], 400);
    }

    $thumbnail = $request->file('thumbnail_image');
    if ($request->hasFile('thumbnail_image')) {
      $thumbnailImage = Uploader::thumbnail_image(public_path(Constant::WEBSITE_ROOM_IMAGE), $thumbnail);
    }

    $room = new Room();
    $room->slider_imgs = json_encode($request['slider_images']);
    $room->featured_img = $thumbnailImage;
    $room->status = $request->status;
    $room->bed = $request->bed;
    $room->bath = $request->bath;
    $room->rent = $request->rent;
    $room->max_guests = $request->max_guests;
    $room->latitude = $request->latitude;
    $room->longitude = $request->longitude;
    $room->address = $request->address;
    $room->phone = $request->phone;
    $room->email = $request->email;
    $room->quantity = $request->quantity;
    $room->user_id = Auth::guard('web')->user()->id;
    $room->category_index = $request->category;
    $room->amenities_index = json_encode($request['amenities']);
    $room->save();

    foreach ($languages as $language) {
      if (
        $request->input($language->code . '_title') ||
        $request->input($language->code . '_summary') ||
        $request->input($language->code . '_description') ||
        $request->input($language->code . '_meta_keywords') ||
        $request->input($language->code . '_meta_description')
      ) {
        $roomContent = new RoomContent();
        $roomContent->language_id = $language->id;
        $roomContent->room_id = $room->id;
        $roomContent->title = $request[$language->code . '_title'];
        $roomContent->slug =   Str::slug($request[$language->code . '_title']);
        $roomContent->summary = Purifier::clean($request[$language->code . '_summary'], 'youtube');
        $roomContent->description = Purifier::clean($request[$language->code . '_description'], 'youtube');
        $roomContent->meta_keywords = $request[$language->code . '_meta_keywords'];
        $roomContent->meta_description = $request[$language->code . '_meta_description'];
        $roomContent->user_id = Auth::guard('web')->user()->id;
        $roomContent->save();
      }
    }

    Session::flash('success', __('Created Successfully'));
    return 'success';
  }

  public function updateFeaturedRoom(Request $request)
  {
    $room = Room::findOrfail($request->roomId);
    if ($request->is_featured == 1) {
      $room->update(['is_featured' => 1]);
    } else {
      $room->update(['is_featured' => 0]);
    }

    Session::flash('success', __('Updated Successfully'));
    return redirect()->back();
  }

  public function editRoom($id)
  {
    $information['languages'] = Language::where('user_id', Auth::guard('web')->user()->id)->get();
    $information['room'] = Room::findOrfail($id);
    return view('user.rooms.edit_room', $information);
  }

  public function updateRoom(Request $request, $id)
  {
    $rules = [
      'category' => 'required',
      'status' => 'required',
      'bed' => 'required',
      'bath' => 'required',
      'rent' => 'required',
      'max_guests' => 'nullable|numeric',
      'quantity' => 'required|numeric',
      'amenities' => 'required',
      'latitude' => 'nullable | numeric| between:-90,90',
      'longitude' => 'nullable | numeric| between:-180,180',
      'thumbnail_image' => 'nullable',
    ];

    $languages = Language::where('user_id', Auth::guard('web')->user()->id)->get();
    $defaulLang = Language::where([['user_id', Auth::guard('web')->user()->id], ['is_default', 1]])->first();
    $languages = Language::where('user_id', Auth::guard('web')->user()->id)->get();

    $rules[$defaulLang->code . '_title'] = 'required|max:255';
    $rules[$defaulLang->code . '_summary'] = 'required';
    $rules[$defaulLang->code . '_description'] = 'required|min:15';

    foreach ($languages as $language) {
      $hasExistingContent = RoomContent::where('room_id', $id)
        ->where('language_id', $language->id)
        ->exists();

      if (
        $hasExistingContent ||
        $request->input($language->code . '_title') ||
        $request->input($language->code . '_summary') ||
        $request->input($language->code . '_description') ||
        $request->input($language->code . '_meta_keywords') ||
        $request->input($language->code . '_meta_description')
      ) {
        $rules[$language->code . '_title'] = 'required';
        $rules[$language->code . '_summary'] = 'required';
        $rules[$language->code . '_description'] = 'required';
      }


      $messages[$language->code . '_title.required'] = __('The title field is required for') . ' ' . $language->name . ' ' . __('language');
      $messages[$language->code . '_title.max'] = __('The title field cannot contain more than 255 characters for') . ' ' . $language->name . ' ' . __('language');

      $messages[$language->code . '_summary.required'] = __('The summary field is required for') . ' ' . $language->name . ' ' . __('language');
      $messages[$language->code . '_description.required'] = __('The description field is required for') . ' ' . $language->name . ' ' . __('language');
      $messages[$language->code . '_description.min'] = __('The description field atleast have 15 characters for') . ' ' . $language->name . ' ' . __('language');
    }

    $validator = Validator::make($request->all(), $rules, $messages);
    if ($validator->fails()) {
      return Response::json([
        'errors' => $validator->getMessageBag()->toArray()
      ], 400);
    }
    $room = Room::findOrFail($id);
    // store featured_img image in storage
    if ($request->hasFile('thumbnail_image')) {
      $newImage = $request->file('thumbnail_image');
      $featuredImage = Uploader::thumbnail_image(public_path(Constant::WEBSITE_ROOM_IMAGE), $newImage, $room->featured_img);
    }

    // merge slider images with existing images if request has new slider image
    if ($request->slider_images) {
      $prevImages = json_decode($room->slider_imgs, true);
      $newImages = $request['slider_images'];
      $imgArr = array_merge($prevImages, $newImages);
    }

    $room->update([
      'slider_imgs' => !empty($imgArr) ? json_encode($imgArr) : $room->slider_imgs,
      'featured_img' => $request->hasFile('thumbnail_image') ? $featuredImage : $room->featured_img,
      'status' => $request->status,
      'bed' => $request->bed,
      'bath' => $request->bath,
      'rent' => $request->rent,
      'max_guests' => $request->max_guests,
      'latitude' => $request->latitude,
      'longitude' => $request->longitude,
      'address' => $request->address,
      'phone' => $request->phone,
      'email' => $request->email,
      'user_id' => Auth::guard('web')->user()->id,
      'quantity' => $request->quantity,
      'category_index' => $request->category,
      'amenities_index' => json_encode($request->amenities),
    ]);

    foreach ($languages as $language) {
      $roomContent = RoomContent::where('room_id', $id)
        ->where('language_id', $language->id)
        ->first();
      $hasExistingContent = RoomContent::where('room_id', $id)
        ->where('language_id', $language->id)
        ->exists();

      if (
        $hasExistingContent ||
        $request->input($language->code . '_title') ||
        $request->input($language->code . '_summary') ||
        $request->input($language->code . '_description') ||
        $request->input($language->code . '_meta_keywords') ||
        $request->input($language->code . '_meta_description')
      ) {
        $content = [
          'language_id' => $language->id,
          'room_id' => $id,
          'title' => $request[$language->code . '_title'],
          'slug' => Str::slug($request[$language->code . '_title']),
          'user_id' => Auth::guard('web')->user()->id,
          'summary' => Purifier::clean($request[$language->code . '_summary'], 'youtube'),
          'description' => Purifier::clean($request[$language->code . '_description'], 'youtube'),
          'meta_keywords' => $request[$language->code . '_meta_keywords'],
          'meta_description' => $request[$language->code . '_meta_description']
        ];


        if (!empty($roomContent)) {
          $roomContent->update($content);
        } else {
          RoomContent::create($content);
        }
      }
    }

    Session::flash('success', __('Updated Successfully'));
    return 'success';
  }

  public function deleteRoom(Request $request)
  {

    $room = Room::findOrFail($request->room_id);
    $roomBooking = RoomBooking::where('room_id', $room->id)->where('user_id', Auth::guard('web')->user()->id)->get();
    if ($room->roomContent()->count() > 0) {
      $contents = $room->roomContent()->get();
      foreach ($contents as $content) {
        $content->delete();
      }
    }

    if ($roomBooking->count() > 0) {
      foreach ($roomBooking as $roomB) {
        // first, delete the attachment
        Uploader::remove(public_path(Constant::WEBSITE_ROOM_BOOKING_ATTACHMENTS), $roomB->attachment);
        // second, delete the invoice
        Uploader::remove(public_path(Constant::WEBSITE_ROOM_BOOKING_INVOICE), $roomB->invoice);
        $roomB->delete();
      }
    }

    if (!is_null($room->slider_imgs)) {
      $images = json_decode($room->slider_imgs);
      foreach ($images as $image) {
        Uploader::remove(public_path(Constant::WEBSITE_ROOM_SLIDER_IMAGE), $image);
      }
    }
    Uploader::remove(public_path(Constant::WEBSITE_ROOM_IMAGE), $room->featured_img);
    $room->delete();

    Session::flash('success', __('Deleted Successfully'));
    return redirect()->back();
  }

  public function bulkDeleteRoom(Request $request)
  {
    $ids = $request->ids;
    foreach ($ids as $id) {
      $room = Room::findOrFail($id);
      $roomBooking = RoomBooking::where('room_id', $room->id)->where('user_id', Auth::guard('web')->user()->id)->get();
      if ($room->roomContent()->count() > 0) {
        $contents = $room->roomContent()->get();
        foreach ($contents as $content) {
          $content->delete();
        }
      }

      if ($roomBooking->count() > 0) {
        foreach ($roomBooking as $roomB) {
          // first, delete the attachment
          Uploader::remove(public_path(Constant::WEBSITE_ROOM_BOOKING_ATTACHMENTS),  $roomB->attachment);
          // second, delete the invoice
          Uploader::remove(public_path(Constant::WEBSITE_ROOM_BOOKING_INVOICE),  $roomB->invoice);
          $roomB->delete();
        }
      }

      if (!is_null($room->slider_imgs)) {
        $images = json_decode($room->slider_imgs);
        foreach ($images as $image) {
          Uploader::remove(public_path(Constant::WEBSITE_ROOM_SLIDER_IMAGE), $image);
        }
      }

      Uploader::remove(public_path(Constant::WEBSITE_ROOM_IMAGE), $room->featured_img);
      $room->delete();
    }

    Session::flash('success', __('Deleted Successfully'));
    return 'success';
  }


  public function bookings(Request $request)
  {
    $booking_number = null;
    if ($request->filled('booking_no')) {
      $booking_number = $request['booking_no'];
    }

    if (URL::current() == Route::is('tenant.room_bookings.all_bookings')) {
      $queryResult['bookings'] = RoomBooking::when($booking_number, function ($query, $booking_number) {
        return $query->where('booking_number', 'like', '%' . $booking_number . '%');
      })->where('user_id', Auth::guard('web')->user()->id)->orderBy('id', 'desc')
        ->paginate(10);
    } else if (URL::current() == Route::is('tenant.room_bookings.paid_bookings')) {
      $queryResult['bookings'] = RoomBooking::when($booking_number, function ($query, $booking_number) {
        return $query->where('booking_number', 'like', '%' . $booking_number . '%');
      })->where('payment_status', 1)
        ->where('user_id', Auth::guard('web')->user()->id)
        ->orderBy('id', 'desc')
        ->paginate(10);
    } else if (URL::current() == Route::is('tenant.room_bookings.unpaid_bookings')) {
      $queryResult['bookings'] = RoomBooking::when($booking_number, function ($query, $booking_number) {
        return $query->where('booking_number', 'like', '%' . $booking_number . '%');
      })->where('payment_status', 0)
        ->where('user_id', Auth::guard('web')->user()->id)
        ->orderBy('id', 'desc')
        ->paginate(10);
    }

    $language = Language::query()->where('user_id', Auth::guard('web')->user()->id)->where('is_default',1)->first();
    $queryResult['roomInfos'] = $language->roomDetails()->whereHas('room', function (Builder $query) {
      $query->where('status', '=', 1)->where('user_id', Auth::guard('web')->user()->id);
    })
      ->select('room_id', 'title')
      ->orderBy('title', 'ASC')
      ->get();
    $queryResult['language']  = $language;
    return view('user.rooms.bookings', $queryResult);
  }

  public function updatePaymentStatus(Request $request)
  {
    $roomBooking = RoomBooking::findOrFail($request->booking_id);
 
    $currentLang = session()->has('lang') ?
      (AdminLanguage::where('code', session()->get('lang'))->first())
      : (AdminLanguage::where('is_default', 1)->first());
    $bs = $currentLang->basic_setting;
    if ($request->payment_status == 1) {
      $roomBooking->update(['payment_status' => 1, 'payment_offline_status' => null]);
    } else {
      $roomBooking->update(['payment_status' => 0, 'payment_offline_status' => null]);
    }
    // delete previous invoice from local storage
    Uploader::remove(public_path(Constant::WEBSITE_ROOM_BOOKING_INVOICE), $roomBooking->invoice);
    // then, generate an invoice in pdf format
    $invoice = $this->generateInvoice($roomBooking);
    // update the invoice field information in database
    $roomBooking->update(['invoice' => $invoice]);
    // finally, send a mail to the customer with the invoice
    $this->sendMailForPaymentStatus($roomBooking, $request->payment_status);

    Session::flash('success', __('Updated Successfully'));
    return redirect()->back();
  }

  public function roomBookingDetails($id)
  {

    $details = RoomBooking::findOrFail($id);
    $queryResult['details'] = $details;
    // get the difference of two dates, date should be in 'YYYY-MM-DD' format
    $date1 = new DateTime($details->arrival_date);
    $date2 = new DateTime($details->departure_date);
    $queryResult['interval'] = $date1->diff($date2, true);
    $language = Language::where('is_default', 1)->where('user_id', Auth::guard('web')->user()->id)->first();

    /**
     * to get the room title first get the room info using eloquent relationship
     * then, get the room content info of that room using eloquent relationship
     * after that, we can access the room title
     * also, get the room category using eloquent relationship
     */
    $roomInfo = $details->hotelRoom()->first();
    $roomContentInfo = $roomInfo->roomContent()->where('language_id', $language->id)->first();
    $queryResult['roomTitle'] = $roomContentInfo->title;

    $roomCategoryName = null;
    if ($roomInfo->roomCategory()) {
      $category = $roomInfo->roomCategory()->where('language_id', $language->id)->first();
      if (!is_null($category)) {
        $roomCategoryName =  $category->name;
      }
    }
    $queryResult['roomCategoryName'] = $roomCategoryName;


    // get all the booked dates of this room
    $roomId = $details->room_id;
    $detailsId = $details->id;
    $queryResult['bookedDates'] = $this->getBookedDatesOfRoom($roomId, $detailsId);
    $queryResult['onlineGateways'] = OnlineGateway::query()
      ->where('status', '=', 1)
      ->select('name')
      ->get();
    $queryResult['offlineGateways'] = OfflineGateway::query()
      ->where('status', '=', 1)
      ->select('name')
      ->orderBy('serial_number', 'asc')
      ->get();
    $queryResult['rent'] = $roomInfo->rent;
    return view('user.rooms.booking_details', $queryResult);
  }

  public function roomBookingEdit($id)
  {
    $details = RoomBooking::findOrFail($id);
    $queryResult['details'] = $details;
    // get the difference of two dates, date should be in 'YYYY-MM-DD' format
    $date1 = new DateTime($details->arrival_date);
    $date2 = new DateTime($details->departure_date);
    $queryResult['interval'] = $date1->diff($date2, true);
    $language = Language::where('is_default', 1)->where('user_id', Auth::guard('web')->user()->id)->first();

    /**
     * to get the room title first get the room info using eloquent relationship
     * then, get the room content info of that room using eloquent relationship
     * after that, we can access the room title
     * also, get the room category using eloquent relationship
     */
    $roomInfo = $details->hotelRoom()->first();
    $roomContentInfo = $roomInfo->roomContent()->where('language_id', $language->id)->first();
    $queryResult['roomTitle'] = $roomContentInfo->title;

    //room category
    $roomCategoryName = null;
    if ($roomInfo->roomCategory()) {
      $category = $roomInfo->roomCategory()->where('language_id', $language->id)->first();
      if (!is_null($category)) {
        $roomCategoryName =  $category->name;
      }
    }
    $queryResult['roomCategoryName'] = $roomCategoryName;
    // get all the booked dates of this room
    $roomId = $details->room_id;
    $detailsId = $details->id;
    $queryResult['bookedDates'] = $this->getBookedDatesOfRoom($roomId, $detailsId);
    $queryResult['onlineGateways'] = OnlineGateway::query()
      ->where('status', '=', 1)
      ->select('name')
      ->get();
    $queryResult['offlineGateways'] = OfflineGateway::query()
      ->where('status', '=', 1)
      ->select('name')
      ->orderBy('serial_number', 'asc')
      ->get();

    $queryResult['rent'] = $roomInfo->rent;
    return view('user.rooms.booking_edit', $queryResult);
  }

  public function updateBooking(AdminRoomBookingRequest $request)
  {
    $currencyInfo = MiscellaneousTrait::getCurrencyInfoUser();
  
    // update the room booking information in database
    $dateArray = explode(' ', $request->dates);
    $onlinePaymentGateway = ['PayPal', 'Stripe', 'Instamojo', 'Paystack', 'Flutterwave', 'Razorpay', 'MercadoPago', 'Mollie', 'Paytm'];
    $gatewayType = in_array($request->payment_method, $onlinePaymentGateway) ? 'online' : 'offline';
    $booking = RoomBooking::query()->findOrFail($request->booking_id);
    $booking->update([
      'customer_name' => $request->customer_name,
      'customer_email' => $request->customer_email,
      'customer_phone' => $request->customer_phone,
      'arrival_date' => $dateArray[0],
      'departure_date' => $dateArray[2],
      'guests' => $request->guests,
      'actual_total' => $request->actual_total,
      'discount' => $request->discount,
      'subtotal' => $request->subtotal,
      'tax' => $request->tax ?? 0.00,
      'fee' => $request->fee ?? 0.00,
      'grand_total' => $request->total,
      'currency_symbol' => $currencyInfo->base_currency_symbol,
      'currency_symbol_position' => $currencyInfo->base_currency_symbol_position,
      'currency_text' => $currencyInfo->base_currency_text,
      'currency_text_position' => $currencyInfo->base_currency_text_position,
      'payment_method' => $request->payment_method,
      'gateway_type' => $gatewayType,
      'payment_status' => $request->payment_status
    ]);
    Uploader::remove(public_path(Constant::WEBSITE_ROOM_BOOKING_INVOICE), $booking->invoice);
    $invoice = $this->generateInvoice($booking);
    // update the invoice field information in database
    $booking->update(['invoice' => $invoice]);

    Session::flash('success', __('Updated Successfully'));
    return 'success';
  }

  public function sendMail(Request $request)
  {
   
    $rules = [
      'subject' => 'required',
      'message' => 'required',
    ];
    $messages = [
      'subject.required' => __('The email subject field is required.'),
      'message.required' => __('The email message field is required.')
    ];
    $validator = Validator::make($request->all(), $rules, $messages);
    if ($validator->fails()) {
      return Response::json([
        'errors' => $validator->getMessageBag()->toArray()
      ], 400);
    }

    $be = BasicExtended::first();
    if (empty($be->smtp_host) || empty($be->smtp_port) || empty($be->encryption) || empty($be->smtp_username) || empty($be->smtp_password)) {
      return back();
    }

    $userBs = BasicSetting::where('user_id', Auth::guard('web')->user()->id)->first();
   

    if ($be->is_smtp == 1) {
      try {
        $smtp = [
          'transport' => 'smtp',
          'host' => $be->smtp_host,
          'port' => $be->smtp_port,
          'encryption' => $be->encryption,
          'username' => $be->smtp_username,
          'password' => $be->smtp_password,
          'timeout' => null,
          'auth_mode' => null,
        ];
        Config::set('mail.mailers.smtp', $smtp);
      } catch (\Exception $e) {
        Session::flash('error', $e->getMessage());
        return back();
      }
    }

    try {
      Mail::send([], [], function ($message) use ($request, $be,$userBs) {
        $fromMail = $be->from_mail;
        $fromName = $userBs->from_name;
        $message->to($request->customer_email)
          ->subject($request->subject)
          ->from($fromMail, $fromName)
          ->replyTo($userBs->reply_to, $userBs->from_name)
          ->html($request->message, 'text/html');
      });
      Session::flash('success', __('Mail sent successfully'));
      return 'success';
    } catch (\Exception $e) {
      Session::flash('error', $e->getMessage());
      return back();
    }
  }

  public function deleteBooking(Request $request, $id)
  {
    $roomBooking = RoomBooking::findOrFail($id);
    // first, delete the attachment
    Uploader::remove(public_path(Constant::WEBSITE_ROOM_BOOKING_ATTACHMENTS), $roomBooking->attachment);
    // second, delete the invoice
    Uploader::remove(public_path(Constant::WEBSITE_ROOM_BOOKING_INVOICE), $roomBooking->invoice);
    // finally, delete the room booking record from db
    $roomBooking->delete();
    Session::flash('success', __('Room booking record deleted successfully!'));
    return redirect()->back();
  }

  public function bulkDeleteBooking(Request $request)
  {
    $ids = $request->ids;
    foreach ($ids as $id) {
      $roomBooking = RoomBooking::findOrFail($id);
      // first, delete the attachment
      Uploader::remove(public_path(Constant::WEBSITE_ROOM_BOOKING_ATTACHMENTS), $roomBooking->attachment);
      // second, delete the invoice
      Uploader::remove(public_path(Constant::WEBSITE_ROOM_BOOKING_INVOICE), $roomBooking->invoice);
      // finally, delete the room booking record from db
      $roomBooking->delete();
    }
    Session::flash('success', __('Deleted Successfully'));
    return 'success';
  }


  private function generateInvoice($bookingInfo)
  {

    $user = Auth::guard('web')->user();
    $fileName = $bookingInfo->booking_number . '.pdf';
    $directory = public_path(Constant::WEBSITE_ROOM_BOOKING_INVOICE . '/');
    @mkdir($directory, 0777, true);
    $language = Language::query()->where('is_default', '=', 1)->where('user_id', '=', $user->id)->first();
    $fileLocated = $directory . $fileName;
    $keywords = get_keywords($user->id);
    Pdf::setOptions([
      'isHtml5ParserEnabled' => true,
      'isRemoteEnabled' => true,
      'logOutputFile' => storage_path('logs/log.htm'),
      'tempDir' => storage_path('logs/'),
    ])->loadView('user.pdf.room_booking', compact('bookingInfo', 'language', 'keywords','user'))->save($fileLocated);
    return $fileName;
  }

  public function pdfTest()
  {
    $bookingInfo = RoomBooking::findOrFail(214);
    $user = Auth::guard('web')->user();
    $language = Language::query()->where('is_default', '=', 1)->where('user_id', '=', $user->id)->first();
    $keywords = get_keywords($user->id);
    $pdf = Pdf::setOptions([
      'isHtml5ParserEnabled' => true,
      'isRemoteEnabled' => true,
      'logOutputFile' => storage_path('logs/log.htm'),
      'tempDir' => storage_path('logs/'),
    ])->loadView('user.pdf.room_booking',compact('bookingInfo', 'language', 'keywords', 'user',));
   
    return $pdf->stream();
  }

  public function pdfPackageTest()
  {
    $bookingInfo = PackageBooking::findOrFail(100);
    $user = Auth::guard('web')->user();
    $language = Language::query()->where('is_default', '=', 1)->where('user_id', '=', $user->id)->first();
    $keywords = get_keywords($user->id);
    $pdf = Pdf::setOptions([
      'isHtml5ParserEnabled' => true,
      'isRemoteEnabled' => true,
      'logOutputFile' => storage_path('logs/log.htm'),
      'tempDir' => storage_path('logs/'),
    ])->loadView('user.pdf.package_booking', compact('bookingInfo', 'language', 'keywords', 'user',));

    return $pdf->stream();
  }

  private function sendMailForPaymentStatus($roomBooking, $status)
  {
    $user = Auth::guard('web')->user();
    if ($status == 1) {
      $mailTemplate = MailTemplate::where('mail_type', 'room_booking_payment_received')->where('user_id', Auth::guard('web')->user()->id)->firstOrFail();
    } else {
      $mailTemplate = MailTemplate::where('mail_type', 'room_booking_payment_cancelled')->where('user_id', Auth::guard('web')->user()->id)->firstOrFail();
    }
    $mailTemplate = str_replace(' ', '_', $mailTemplate->mail_type);
    $infoCus = DB::table('user_basic_settings')->where('user_id', '=', Auth::guard('web')->user()->id)->first();
    $mailer = new MegaMailer;
    $data = [
      'toMail' => $roomBooking->customer_email,
      'customer_name' => $roomBooking->customer_name,
      'website_title' => $infoCus->website_title,
      'templateType' => $mailTemplate,
      'type' => $mailTemplate
    ];
    $mailer->mailFromUser($data, $user);
  }

  // room booking from admin panel
  public function bookedDates(Request $request)
  {
    $rule = [
      'room_id' => 'required'
    ];
    $message = [
      'room_id.required' => __('Please select a room.')
    ];

    $validator = Validator::make($request->all(), $rule, $message);
    if ($validator->fails()) {
      return response()->json([
        'error' => $validator->getMessageBag()
      ]);
    }
    // get all the booked dates of the selected room
    $roomId = $request['room_id'];
    $bookedDates = $this->getBookedDatesOfRoom($roomId);
    Session::put('bookedDates', $bookedDates);
    return response()->json([
      'success' => route('tenant.room_bookings.booking_form', ['room_id' => $roomId])
    ]);
  }

  public function getBookedDatesOfRoom($id, $bookingId = null)
  {
    $quantity = Room::query()->findOrFail($id)->quantity;
    $bookings = RoomBooking::query()->where('room_id', '=', $id)
      ->where('payment_status', '=', 1)
      ->select('arrival_date', 'departure_date')
      ->get();
    $bookedDates = [];
    foreach ($bookings as $booking) {
      // get all the dates between the booking arrival date & booking departure date
      $date_1 = $booking->arrival_date;
      $date_2 = $booking->departure_date;
      $allDates = $this->getAllDates($date_1, $date_2, 'Y-m-d');
      // loop through the list of dates, which we have found from the booking arrival date & booking departure date
      foreach ($allDates as $date) {
        $bookingCount = 0;
        // loop through all the bookings
        foreach ($bookings as $currentBooking) {
          $bookingStartDate = Carbon::parse($currentBooking->arrival_date);
          $bookingEndDate = Carbon::parse($currentBooking->departure_date);
          $currentDate = Carbon::parse($date);
          // check for each date, whether the date is present or not in any of the booking date range
          if ($currentDate->betweenIncluded($bookingStartDate, $bookingEndDate)) {
            $bookingCount++;
          }
        }

        // if the number of booking of a specific date is same as the room quantity, then mark that date as unavailable
        if ($bookingCount >= $quantity && !in_array($date, $bookedDates)) {
          array_push($bookedDates, $date);
        }
      }
    }

    if (is_null($bookingId)) {
      return $bookedDates;
    } else {
      $booking = RoomBooking::query()->findOrFail($bookingId);
      $arrivalDate = $booking->arrival_date;
      $departureDate = $booking->departure_date;
      // get all the dates between the booking arrival date & booking departure date
      $bookingAllDates = $this->getAllDates($arrivalDate, $departureDate, 'Y-m-d');

      // remove dates of this booking from 'bookedDates' array while editing a room booking
      foreach ($bookingAllDates as $date) {
        $key = array_search($date, $bookedDates);
        if ($key !== false) {
          unset($bookedDates[$key]);
        }
      }
      return array_values($bookedDates);
    }
  }

  public function getAllDates($startDate, $endDate, $format)
  {
    $dates = [];
    // convert string to timestamps
    $currentTimestamps = strtotime($startDate);
    $endTimestamps = strtotime($endDate);
    // set an increment value
    $stepValue = '+1 day';
    // push all the timestamps to the 'dates' array by formatting those timestamps into date
    while ($currentTimestamps <= $endTimestamps) {
      $formattedDate = date($format, $currentTimestamps);
      array_push($dates, $formattedDate);
      $currentTimestamps = strtotime($stepValue, $currentTimestamps);
    }
    return $dates;
  }

  public function bookingForm(Request $request)
  {
    if (Session::has('bookedDates')) {
      $queryResult['dates'] = Session::get('bookedDates');
    } else {
      $queryResult['dates'] = [];
    }
    $id = $request['room_id'];
    $queryResult['id'] = $id;
    $room = Room::query()->find($id);
    $queryResult['rent'] = $room->rent;
    $queryResult['currencyInfo'] = MiscellaneousTrait::getCurrencyInfoUser();
    $queryResult['onlineGateways'] = OnlineGateway::query()
      ->where('status', '=', 1)->where('user_id', Auth::guard('web')->user()->id)
      ->select('name')
      ->get();
    $queryResult['offlineGateways'] = OfflineGateway::query()
      ->where('status', '=', 1)->where('user_id', Auth::guard('web')->user()->id)
      ->select('name')
      ->orderBy('serial_number', 'asc')
      ->get();

    return view('user.rooms.booking_form', $queryResult);
  }

  public function makeBooking(AdminRoomBookingRequest  $request)
  {
    $currencyInfo = MiscellaneousTrait::getCurrencyInfoUser();
    // store the room booking information in database
    $dateArray = explode(' ', $request->dates);

    $onlinePaymentGateway = ['PayPal', 'Stripe', 'Instamojo', 'Paystack', 'Flutterwave', 'Razorpay', 'MercadoPago', 'Mollie', 'Paytm'];
    $gatewayType = in_array($request->payment_method, $onlinePaymentGateway) ? 'online' : 'Offline';

    $bookingInfo = RoomBooking::query()->create([
      'booking_number' => time(),
      'user_id' => Auth::guard('web')->user()->id,
      'customer_name' => $request->customer_name,
      'customer_email' => $request->customer_email,
      'customer_phone' => $request->customer_phone,
      'room_id' => $request->room_id,
      'arrival_date' => $dateArray[0],
      'departure_date' => $dateArray[2],
      'guests' => $request->guests,
      'actual_total' => $request->actual_total,
      'discount' => $request->discount,
      'subtotal' => $request->subtotal,
      'tax' => $request->tax ?? 0.00,
      'fee' => $request->fee ?? 0.00,
      'grand_total' => $request->total,
      'currency_symbol' => $currencyInfo->base_currency_symbol,
      'currency_symbol_position' => $currencyInfo->base_currency_symbol_position,
      'currency_text' => $currencyInfo->base_currency_text,
      'currency_text_position' => $currencyInfo->base_currency_text_position,
      'payment_method' => $request->payment_method,
      'gateway_type' => $gatewayType,
      'payment_status' => $request->payment_status
    ]);

    if ($request->payment_status == 1) {
      // generate an invoice in pdf format
      $invoice = $this->generateInvoice($bookingInfo);
      // update the invoice field information in database
      $bookingInfo->update(['invoice' => $invoice]);
      // send a mail to the customer with an invoice
      $this->sendMailForRoomBooking($bookingInfo);
    }

    session()->flash('success', 'Room has booked.');
    return 'success';
  }

  public function sendMailForRoomBooking($bookingInfo)
  {
    $infoCus = DB::table('user_basic_settings')->first();
    // get the difference of two dates, date should be in 'YYYY-MM-DD' format
    $date1 = new DateTime($bookingInfo->arrival_date);
    $date2 = new DateTime($bookingInfo->departure_date);
    $interval = $date1->diff($date2, true);

    // get the room category name according to language
    $language = Language::query()->where('is_default', '=', 1)->where('user_id', '=', Auth::guard('web')->user()->id)->first();

    $roomContent = RoomContent::query()->where('language_id', '=', $language->id)
      ->where('room_id', '=', $bookingInfo->room_id)
      ->first();

    $room = Room::where('id', $bookingInfo->room_id)->first();
    $roomCategory = $room->roomCategory()->where('language_id', $language->id)->first();

    $roomCategoryName = !is_null($roomCategory) ? $roomCategory->name : '';

    $roomRent = ($bookingInfo->currency_text_position == 'left' ? $bookingInfo->currency_text . ' ' : '') . $bookingInfo->grand_total . ($bookingInfo->currency_text_position == 'right' ? ' ' . $bookingInfo->currency_text : '');

    // get the amenities of booked room
    $amenityIds = json_decode($room->amenities_index);

    $amenityArray = [];
    foreach ($amenityIds as $id) {
      $amenity = RoomAmenity::where('indx', $id)->where('language_id', $language->id)->first();
      if (!is_null($amenity)) {
        array_push($amenityArray, $amenity->name);
      }
    }

    // now, convert amenity array into comma separated string
    $amenityString = implode(', ', $amenityArray);

    // replace template's curly-brace string with actual data
    $mailer = new MegaMailer();
    $data = [
      'toMail' => $bookingInfo->customer_email,
      'toName' =>  $bookingInfo->customer_name,
      'customer_name' => $bookingInfo->customer_name,
      'room_name' => $roomContent->title,
      'room_rent' => $roomRent,
      'booking_number' => $bookingInfo->booking_number,
      'booking_date' =>  date_format($bookingInfo->created_at, 'F d, Y'),
      'number_of_night' =>  $interval->days,
      'website_title' => $infoCus->website_title,
      'check_in_date' => $bookingInfo->arrival_date,
      'discount' => $bookingInfo->discount ?? 0,
      'check_out_date' => $bookingInfo->departure_date,
      'number_of_guests' => $bookingInfo->guests,
      'room_type' => $roomCategoryName,
      'room_amenities' => $amenityString,
      'invoice' => public_path(Constant::WEBSITE_ROOM_BOOKING_INVOICE . '/' . $bookingInfo->invoice),
      'templateType' => $bookingInfo->gateway_type == 'Offline' ? 'room_booking_for_offline_gateway' : 'room_booking_for_online_gateway',
      'type' => 'roombooking'
    ];

    $mailer->mailFromUser($data, Auth::guard('web')->user());
  }


  public function report(Request $request)
  {
    $queryResult['onlineGateways'] = OnlineGateway::query()->where('status', '=', 1)->where('user_id', Auth::guard('web')->user()->id)->get();
    $queryResult['offlineGateways'] = OfflineGateway::query()->where('status', '=', 1)->where('user_id', Auth::guard('web')->user()->id)->orderBy('serial_number', 'asc')->get();

    $from = $to = $paymentGateway = $paymentStatus = null;
    if ($request->filled('payment_gateway')) {
      $paymentGateway = $request->payment_gateway;
    }
    if ($request->filled('payment_status')) {
         $paymentStatus = $request->payment_status ; 
    }


    if ($request->filled('from') && $request->filled('to')) {
      $from = Carbon::parse($request->from)->toDateString();
      $to = Carbon::parse($request->to)->toDateString();

      $records = RoomBooking::query()
        ->where('user_id', Auth::guard('web')->user()->id)
        ->whereDate('created_at', '>=', $from)
        ->whereDate('created_at', '<=', $to)
        ->when($paymentGateway, function (Builder $query, $paymentGateway) {
          return $query->where('payment_method', '=', $paymentGateway);
        })
        ->when($paymentStatus, function (Builder $query, $paymentStatus) {
          $status =  $paymentStatus == 'incompleted' ? 0 : 1;
          return $query->where('payment_status', '=', $status);
        })
        ->select('*')->orderByDesc('id');

      $collection_1 = $this->manipulateCollection($records->get());
      Session::put('room_bookings', $collection_1);

      $collection_2 = $this->manipulateCollection($records->paginate(10));
      $queryResult['bookings'] = $collection_2;
    } else {
      Session::put('room_bookings', null);
      $queryResult['bookings'] = [];
    }
    return view('user.rooms.report', $queryResult);
  }

  public function manipulateCollection($bookings)
  {

    $bookings->map(function ($booking) {
      // room title
      $room = $booking->hotelRoom()->where('user_id', Auth::guard('web')->user()->id)->first();
      $booking['roomTitle'] = $room->roomContent()->where('room_id', $room->id)->pluck('title')->first();
      // format booking start date
      $startDateObj = Carbon::parse($booking->start_date);
      $booking['startDate'] = $startDateObj->format('M d, Y');

      // format booking end date
      $endDateObj = Carbon::parse($booking->end_date);
      $booking['endDate'] = $endDateObj->format('M d, Y');

      // format booking create date
      $createDateObj = Carbon::parse($booking->created_at);
      $booking['createdAt'] = $createDateObj->format('M d, Y');
    });

    return $bookings;
  }

  public function exportReport()
  {
    if (Session::has('room_bookings')) {
      $roomBooking = Session::get('room_bookings');

      if (count($roomBooking) == 0) {
        Session::flash('warning', 'No booking found to export!');
        return redirect()->back();
      } else {
        return Excel::download(new RoomBookingsExport($roomBooking), 'room-bookings.csv');
      }
    } else {
      Session::flash('error', __('There has no booking to export.'));
      return redirect()->back();
    }
  }

  public function generateValidationRules(Request $request, $languages, $fields)
  {
    $rules = [];

    foreach ($languages as $language) {
      $prefix = $language->code . '_';

      if ($language->default == 1) {
        // If the language is default, all fields are required
        foreach ($fields as $field) {
          $rules[$prefix . $field] = 'required';
        }
      } else {
        // For other languages, check if any field is filled
        $hasInput = false;
        foreach ($fields as $field) {
          if ($request->input($prefix . $field)) {
            $hasInput = true;
            break;
          }
        }

        // If any field has input, make all fields required for that language
        if ($hasInput) {
          foreach ($fields as $field) {
            $rules[$prefix . $field] = 'required';
          }
        }
      }
    }

    return $rules;
  }

  public function taxFee(Request $request)
  {
    $data['data'] = BasicSetting::select('room_tax_status', 'room_tax', 'room_fee_status', 'room_fee', 'package_tax', 'package_tax_status', 'package_fee_status', 'package_fee', 'base_currency_symbol')->where('user_id', Auth::guard('web')->user()->id)->first();

    return view('user.rooms.tex_fee', $data);
  }
  public function updateTaxFee(Request $request)
  {
    $rules = [
      'room_tax_status' => 'required',
      'room_tax' => 'required_if:room_tax_status,1',
      'room_fee_status' => 'required',
      'room_fee' => 'required_if:room_fee_status,1',
    ];

    $validator = Validator::make($request->all(), $rules);
    if ($validator->fails()) {
      return response()->json(['errors' => $validator->errors()], 422);
    }
    BasicSetting::query()->updateOrInsert(
      ['user_id' => Auth::guard('web')->user()->id],
      $request->except(['_token', 'user_id'] + [
        'user_id' => Auth::guard('web')->user()->id,
      ])
    );
    Session::flash('success', __('Updated Successfully'));
    return "success";
  }
}
