<?php

namespace App\Http\Controllers\User;

use App\Constants\Constant;
use App\Enums\ThemeVersion;
use Carbon\Carbon;
use Illuminate\Support\Str;
use App\Models\User\Package;
use Illuminate\Http\Request;
use App\Models\BasicExtended;
use App\Models\User\Language;
use App\Http\Helpers\MegaMailer;
use App\Models\User\PackagePlan;
use App\Models\User\MailTemplate;
use App\Models\User\PackageCoupon;
use App\Traits\MiscellaneousTrait;
use Illuminate\Support\Facades\DB;
use App\Models\User\OfflineGateway;
use App\Models\User\PackageBooking;
use App\Models\User\PackageContent;
use Illuminate\Support\Facades\URL;
use Mews\Purifier\Facades\Purifier;
use App\Http\Controllers\Controller;
use App\Http\Requests\CouponRequest;
use App\Models\User\PackageCategory;
use App\Models\User\PackageLocation;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Mail;
use Maatwebsite\Excel\Facades\Excel;
use Illuminate\Support\Facades\Route;
use App\Exports\PackageBookingsExport;
use App\Http\Helpers\Uploader;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Response;
use App\Models\Language as AdminLanguage;
use App\Models\User\BasicSetting;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\Validator;
use App\Models\User\PaymentGateway as OnlineGateway;
use App\Rules\ImageMimeTypeRule;
use Barryvdh\DomPDF\Facade\Pdf;

class PackageController extends Controller
{
  use MiscellaneousTrait;

  public function settings()
  {
    $data = DB::table('user_basic_settings')
      ->select('package_rating_status', 'package_guest_checkout_status')
      ->where('uniqid', 12345)
      ->first();

    return view('user.packages.settings', ['data' => $data]);
  }

  public function updateSettings(Request $request)
  {
    $rules = [
      'package_rating_status' => 'required',
      'package_guest_checkout_status' => 'required'
    ];
    $validator = Validator::make($request->all(), $rules);
    if ($validator->fails()) {
      return redirect()->back()->withErrors($validator->errors());
    }
    DB::table('user_basic_settings')->update([
      'package_rating_status' => $request->package_rating_status,
      'package_guest_checkout_status' => $request->package_guest_checkout_status
    ]);

    Session::flash('success', __('Updated Successfully'));
    return redirect()->back();
  }

  public function coupons()
  {
    // get the PackageCoupon from db
    $information['coupons'] = PackageCoupon::where('user_id', Auth::guard('web')->user()->id)->orderByDesc('id')->get();
    // also, get the currency information from db
    $information['currencyInfo'] = MiscellaneousTrait::getCurrencyInfoUser();
    $language = Language::where('user_id', Auth::guard('web')->user()->id)->first();
    $packages = Package::where('user_id', Auth::guard('web')->user()->id)->orderByDesc('id')->get();
    $packages->map(function ($package) use ($language) {
      $package['title'] = $package->packageContent()->where('language_id', $language->id)->where('user_id', Auth::guard('web')->user()->id)->pluck('title')->first();
    });
    $information['packages'] = $packages;

    return view('user.packages.coupons', $information);
  }

  public function storeCoupon(CouponRequest $request)
  {
    $startDate = Carbon::parse($request->start_date);
    $endDate = Carbon::parse($request->end_date);
    if ($request->filled('packages')) {
      $packages = $request->packages;
    }
    PackageCoupon::create($request->except('start_date', 'end_date', 'packages') + [
      'start_date' => date_format($startDate, 'Y-m-d'),
      'end_date' => date_format($endDate, 'Y-m-d'),
      'user_id' => Auth::guard('web')->user()->id,
      'packages' => isset($packages) ? json_encode($packages) : null
    ]);

    Session::flash('success', __('Created Successfully'));
    return 'success';
  }

  public function updateCoupon(CouponRequest $request)
  {
    $startDate = Carbon::parse($request->start_date);
    $endDate = Carbon::parse($request->end_date);
    if ($request->filled('packages')) {
      $packages = $request->packages;
    }
    PackageCoupon::find($request->id)->update($request->except('start_date', 'end_date', 'packages') + [
      'start_date' => date_format($startDate, 'Y-m-d'),
      'end_date' => date_format($endDate, 'Y-m-d'),
      'packages' => isset($packages) ? json_encode($packages) : null
    ]);

    Session::flash('success', __('Updated Successfully'));
    return 'success';
  }

  public function destroyCoupon($id)
  {
    PackageCoupon::find($id)->delete();
    return redirect()->back()->with('success', __('Deleted Successfully'));
  }


  public function categories(Request $request)
  {
    // first, get the language info from db
    $language = Language::where('code', $request->language)->where('user_id', Auth::guard('web')->user()->id)->firstOrFail();
    $information['language'] = $language;
    // then, get the package categories of that language from db
    $information['packageCategories'] = PackageCategory::where('language_id', $language->id)->where('user_id', Auth::guard('web')->user()->id)
      ->orderBy('id', 'desc')
      ->paginate(10);
    // also, get all the languages from db
    $information['langs'] = Language::where('user_id', Auth::guard('web')->user()->id)->get();
    return view('user.packages.categories', $information);
  }

  public function storeCategory(Request $request)
  {
    $settings = BasicSetting::select('theme_version')->where('user_id', Auth::guard('web')->user()->id)->first();
    $rules = [
      'status' => 'required',
      'serial_number' => 'required'
    ];
    $languages = Language::where('user_id', Auth::guard('web')->user()->id)->get();
    $messages = [];
    foreach ($languages as $language) {
      $slug = slug_create($request[$language->code . '_name']);
      $rules[$language->code . '_name'] =
        [
          'required',
          'max:255',
          function ($attribute, $value, $fail) use ($slug, $language) {
            $bis = PackageCategory::where('language_id', $language->id)->where('user_id', Auth::guard('web')->user()->id)->get();
            foreach ($bis as $key => $bi) {
              if (strtolower($slug) == strtolower($bi->slug)) {
                $fail('The name field must be unique for ' . $language->name . ' language.');
              }
            }
          }
        ];
      $messages[$language->code . '_name.required'] = __('The name field is required for') . ' ' . $language->name . ' ' . __('language');
    }

    if ($settings->theme_version == ThemeVersion::ThemeFour->value) {
      $rules['category_image'] = [
        'required',
        new ImageMimeTypeRule()
      ];
    }
    $validator = Validator::make($request->all(), $rules, $messages);
    if ($validator->fails()) {
      return Response::json([
        'errors' => $validator->getMessageBag()->toArray()
      ], 400);
    }
    $category_image = null;
    if ($request->hasFile('category_image')) {
      $category_image = Uploader::upload_picture(
        public_path(Constant::WEBSITE_PACKAGE_IMAGE),
        $request->file('category_image'),
      );
    }
    $uid = uniqid();
    foreach ($languages as $language) {
      $PackageCategory = new PackageCategory();
      $PackageCategory->name = $request[$language->code . '_name'];
      $PackageCategory->slug = slug_create($request[$language->code . '_name']);
      $PackageCategory->language_id  = $language->id;
      $PackageCategory->user_id  = Auth::guard('web')->user()->id;
      $PackageCategory->indx  = $uid;
      $PackageCategory->image  = isset($category_image) ? $category_image : null;
      $PackageCategory->status  = $request->status;
      $PackageCategory->serial_number  = $request->serial_number;
      $PackageCategory->save();
    }

    Session::flash('success', __('Created Successfully'));
    return 'success';
  }

  public function editCategory($id)
  {
    $userId = Auth::guard('web')->user()->id;
    $data['languages'] = Language::where('user_id', $userId)->get();
    $data['data'] = PackageCategory::query()
      ->where('user_id', $userId)
      ->findorFail($id);

    return view('user.packages.edit_category', $data);
  }

  public function updateCategory(Request $request)
  {
    $settings = BasicSetting::select('theme_version')->where('user_id', Auth::guard('web')->user()->id)->first();
    $rules = [
      'status' => 'required',
      'serial_number' => 'required'
    ];
    $messages = [];
    $languages = Language::where('user_id', Auth::guard('web')->user()->id)->get();
    foreach ($languages as $language) {
      $slug = slug_create($request[$language->code . '_name']);
      $indx = $request->category_indx;
      $rules[$language->code . '_name'] =
        [
          'required',
          'max:255',
          function ($attribute, $value, $fail) use ($slug, $indx, $language) {
            $bis = PackageCategory::where('indx', '<>', $indx)
              ->where('language_id', $language->id)
              ->where('user_id', Auth::guard('web')->user()->id)
              ->get();
            foreach ($bis as $key => $bi) {
              if (strtolower($slug) == strtolower($bi->slug)) {
                $fail('The name field must be unique for ' . $language->name . ' language.');
              }
            }
          }
        ];
      $messages[$language->code . '_name.required'] = __('The name field is required for') . ' ' . $language->name . ' ' . __('language');
    }
    if ($settings->theme_version == ThemeVersion::ThemeFour->value) {
      $rules['category_image'] = [
        'nullable',
        new ImageMimeTypeRule()
      ];
    }
    $validator = Validator::make($request->all(), $rules, $messages);
    if ($validator->fails()) {
      return Response::json([
        'errors' => $validator->getMessageBag()->toArray()
      ], 400);
    }


    $category_image = null;
    if ($request->hasFile('category_image')) {
      $category_image = Uploader::upload_picture(public_path(Constant::WEBSITE_PACKAGE_IMAGE), $request->file('category_image'));
    }


    foreach ($languages as $language) {
      $PackageCategory = PackageCategory::query()
        ->where('user_id', Auth::guard('web')->user()->id)
        ->where('indx', $request->category_indx)->where('language_id', $language->id)->first();

      if ($settings->theme_version == ThemeVersion::ThemeFour->value && !empty($category_image)) {
        Uploader::remove(public_path(Constant::WEBSITE_PACKAGE_IMAGE), $PackageCategory->image);
      }
      
      $PackageCategory->name = $request[$language->code . '_name'];
      $PackageCategory->slug = slug_create($request[$language->code . '_name']);
      $PackageCategory->language_id  = $language->id;
      $PackageCategory->user_id  = Auth::guard('web')->user()->id;
      $PackageCategory->status  = $request->status;
      $PackageCategory->serial_number  = $request->serial_number;
      $PackageCategory->image  = !empty($category_image) ? $category_image :  $PackageCategory->image;
      $PackageCategory->update();
    }

    Session::flash('success', __('Updated Successfully'));
    return 'success';
  }

  public function deleteCategory(Request $request)
  {
    $packageCategory = PackageCategory::findOrFail($request->category_id);

    if ($packageCategory->packageContentList()->count() > 0) {
      Session::flash('warning', __('First delete all the packages of this category!'));
      return redirect()->back();
    }

    $packageCategory->delete();
    Session::flash('success', __('Deleted Successfully'));
    return redirect()->back();
  }

  public function bulkDeleteCategory(Request $request)
  {
    $ids = $request->ids;
    foreach ($ids as $id) {
      $packageCategory = PackageCategory::findOrFail($id);

      if ($packageCategory->packageContentList()->count() > 0) {
        Session::flash('warning', __('First delete all the packages of those category!'));

        return 'success';
      }
      $packageCategory->delete();
    }

    Session::flash('success', __('Deleted Successfully'));
    return 'success';
  }


  public function packages()
  {
    $language = Language::where('user_id', Auth::guard('web')->user()->id)->firstOrFail();
    $information['language'] = $language;
    $information['packageContents'] = PackageContent::with('package')
      ->where('language_id', '=', $language->id)
      ->where('user_id', Auth::guard('web')->user()->id)
      ->orderBy('package_id', 'desc')
      ->get();
    $information['currencyInfo'] = MiscellaneousTrait::getCurrencyInfoUser();

    return view('user.packages.packages', $information);
  }

  public function createPackage()
  {
    // get all the languages from db
    $information['languages'] = Language::where('user_id', Auth::guard('web')->user()->id)->get();
    $information['basicSettings'] = DB::table('user_basic_settings')
      ->where('uniqid', 12345)
      ->first();

    return view('user.packages.create_package', $information);
  }

  public function storePackage(Request $request)
  {

    $rules = [
      'number_of_days' => 'required|numeric|min:1',
      'plan_type' => 'required',
      'category' => 'required',
      'pricing_type' => 'required',
      'thumbnail_image' => 'required',
      'slider_images' => 'required|max:5000',
      'fixed_package_price' => 'required_if:pricing_type,==,fixed',
      'per_person_package_price' => 'required_if:pricing_type,==,per-person',
    ];
    $defaulLang = Language::where([['user_id', Auth::guard('web')->user()->id], ['is_default', 1]])->first();
    $languages = Language::where('user_id', Auth::guard('web')->user()->id)->get();
    $settings = DB::table('user_basic_settings')
      ->where('uniqid', 12345)
      ->first();

    //default language validation
    $rules[$defaulLang->code . '_title'] = 'required|max:255';
    $rules[$defaulLang->code . '_description'] = 'required|min:15';

    foreach ($languages as $language) {
      if (
        $request->input($language->code . '_title') ||
        $request->input($language->code . '_description') ||
        $request->input($language->code . '_meta_keywords') ||
        $request->input($language->code . '_meta_description')
      ) {
        $rules[$language->code . '_title'] = 'required';
        $rules[$language->code . '_description'] = 'required';
      }

      $messages[$language->code . '_title.required'] = __('The title field is required for') . ' ' . $language->name . ' ' . __('language');

      $messages[$language->code . '_title.max'] = __('The title field cannot contain more than 255 characters for') . ' ' . $language->name . ' ' . __('language');

      $messages[$language->code . '_description.required'] = __('The description field is required for') . ' ' . $language->name . ' ' . __('language');

      $messages[$language->code . '_description.min'] = __('The description field atleast have 15 characters for') . ' ' . $language->name . ' ' . __('language');
    }

    $validator = Validator::make($request->all(), $rules, $messages);

    if ($validator->fails()) {
      return Response::json([
        'errors' => $validator->getMessageBag()->toArray()
      ], 400);
    }

    $thumbnail = $request->file('thumbnail_image');
    if ($request->hasFile('thumbnail_image')) {
      $thumbnailImage = Uploader::thumbnail_image_2(public_path(Constant::WEBSITE_PACKAGE_IMAGE), $thumbnail);
    }

    $package = new Package();
    $package->slider_imgs = json_encode($request['slider_images']);
    $package->featured_img = $thumbnailImage ?? '';
    $package->user_id = Auth::guard('web')->user()->id;
    $package->plan_type = $request->plan_type;
    $package->number_of_days = $request->number_of_days;
    $package->max_persons = $request->max_persons;
    $package->pricing_type = $request->pricing_type;
    if ($request->pricing_type == 'fixed') {
      $package->package_price = $request->fixed_package_price;
    } elseif ($request->pricing_type == 'per-person') {
      $package->package_price = $request->per_person_package_price;
    }
    $package->email = $request->email;
    $package->phone = $request->phone;
    $package->package_category_index = $request->category;


    $package->save();
    foreach ($languages as $language) {
      if (
        $request->input($language->code . '_title') ||
        $request->input($language->code . '_description') ||
        $request->input($language->code . '_meta_keywords') ||
        $request->input($language->code . '_meta_description')
      ) {
        $packageContent = new PackageContent();
        $packageContent->language_id = $language->id;
        $packageContent->user_id = Auth::guard('web')->user()->id;
        $packageContent->package_id = $package->id;
        $packageContent->title = $request[$language->code . '_title'];
        $packageContent->slug = Str::slug($request[$language->code . '_title']);
        $packageContent->description = Purifier::clean($request[$language->code . '_description'], 'youtube');
        $packageContent->meta_keywords = $request[$language->code . '_meta_keywords'];
        $packageContent->meta_description = $request[$language->code . '_meta_description'];
        $packageContent->save();
      }
    }

    Session::flash('success', __('Created Successfully'));
    return 'success';
  }

  public function updateFeaturedPackage(Request $request)
  {
    $package = Package::findOrFail($request->packageId);
    if ($request->is_featured == 1) {
      $package->update(['is_featured' => 1]);
    } else {
      $package->update(['is_featured' => 0]);
    }

    Session::flash('success', __('Updated Successfully'));
    return redirect()->back();
  }

  public function editPackage($id)
  {
    $information['package'] = Package::findOrFail($id);
    // get all the languages from db
    $information['languages'] = Language::where('user_id', Auth::guard('web')->user()->id)->get();
    $information['basicSettings'] = DB::table('user_basic_settings')
      ->where('uniqid', 12345)
      ->first();

    return view('user.packages.edit_package', $information);
  }

  public function getSliderImages($id)
  {
    $package = Package::findOrFail($id);
    $sliderImages = json_decode($package->slider_imgs);
    $images = [];
    // concatanate slider image with image location
    foreach ($sliderImages as $key => $sliderImage) {
      $data = url(public_path(Constant::WEBSITE_PACKAGE_SLIDER_IMAGE  . '/' . $sliderImage));
      array_push($images, $data);
    }

    return Response::json($images, 200);
  }

  public function updatePackage(Request $request, $id)
  {

    $rules = [
      'number_of_days' => 'required|numeric|min:1',
      'category' => 'required',
      'plan_type' => 'required',
      'pricing_type' => 'required',
      'fixed_package_price' => 'required_if:pricing_type,==,fixed',
      'per_person_package_price' => 'required_if:pricing_type,==,per-person',
      'thumbnail_image' => 'nullable',
    ];

    $languages = Language::where('user_id', Auth::guard('web')->user()->id)->get();
    $defaulLang = Language::where([['user_id', Auth::guard('web')->user()->id], ['is_default', 1]])->first();

    $rules[$defaulLang->code . '_title'] = 'required|max:255';
    $rules[$defaulLang->code . '_description'] = 'required|min:15';

    foreach ($languages as $language) {
      $hasExistingContent = PackageContent::where('package_id', $id)
        ->where('language_id', $language->id)
        ->exists();

      if (
        $hasExistingContent ||
        $request->input($language->code . '_title') ||
        $request->input($language->code . '_description') ||
        $request->input($language->code . '_meta_keywords') ||
        $request->input($language->code . '_meta_description')
      ) {
        $rules[$language->code . '_title'] = 'required';
        $rules[$language->code . '_description'] = 'required';
      }


      $messages[$language->code . '_title.required'] = __('The title field is required for') . ' ' . $language->name . ' ' . __('language');
      $messages[$language->code . '_title.max'] = __('The title field cannot contain more than 255 characters for') . ' ' . $language->name . ' ' . __('language');
      $messages[$language->code . '_description.required'] = __('The description field is required for') . ' ' . $language->name . ' ' . __('language');
      $messages[$language->code . '_description.min'] = __('The description field atleast have 15 characters for') . ' ' . $language->name . ' ' . __('language');
    }

    $validator = Validator::make($request->all(), $rules, $messages);
    if ($validator->fails()) {
      return Response::json([
        'errors' => $validator->getMessageBag()->toArray()
      ], 400);
    }

    $package = Package::findOrFail($id);

    // store featured_img image in storage
    if ($request->hasFile('thumbnail_image')) {
      $newImage = $request->file('thumbnail_image');
      $oldImage = $package->featured_img;
      $featuredImage = Uploader::thumbnail_image_2(public_path(Constant::WEBSITE_PACKAGE_IMAGE), $newImage, $oldImage);
    }

    // merge slider images with existing images if request has new slider image
    if ($request->slider_images) {
      $prevImages = json_decode($package->slider_imgs, true);
      $newImages = $request['slider_images'];
      $imgArr = array_merge($prevImages, $newImages);
    }
    // get the package price that admin has selected
    if ($request->pricing_type == 'negotiable') {
      $amount = null;
    } elseif ($request->pricing_type == 'fixed') {
      $amount = $request->fixed_package_price;
    } elseif ($request->pricing_type == 'per-person') {
      $amount = $request->per_person_package_price;
    }

    $package->update([
      'number_of_days' => $request->number_of_days,
      'slider_imgs' => isset($imgArr) ? json_encode($imgArr) : $package->slider_imgs,
      'featured_img' => $request->hasFile('thumbnail_image') ? $featuredImage : $package->featured_img,
      'plan_type' => $request->plan_type,
      'max_persons' => $request->max_persons,
      'pricing_type' => $request->pricing_type,
      'package_price' => isset($amount) ? $amount : $package->package_price,
      'email' => $request->email,
      'phone' => $request->phone,
      'package_category_index' => $request->category,
    ]);

    foreach ($languages as $language) {

      $packageContent = PackageContent::where('package_id', $id)
        ->where('user_id', Auth::guard('web')->user()->id)
        ->where('language_id', $language->id)
        ->first();

      $hasExistingContent = PackageContent::where('package_id', $id)
        ->where('language_id', $language->id)
        ->exists();

      if (
        $hasExistingContent ||
        $request->input($language->code . '_title') ||
        $request->input($language->code . '_description') ||
        $request->input($language->code . '_meta_keywords') ||
        $request->input($language->code . '_meta_description')
      ) {


        $content = [
          'language_id' => $language->id,
          'package_id' => $package->id,
          'user_id' => Auth::guard('web')->user()->id,
          'title' => $request[$language->code . '_title'],
          'slug' => Str::slug($request[$language->code . '_title']),
          'description' => Purifier::clean($request[$language->code . '_description'], 'youtube'),
          'meta_keywords' => $request[$language->code . '_meta_keywords'],
          'meta_description' => $request[$language->code . '_meta_description']
        ];

        if (!empty($packageContent)) {
          $packageContent->update($content);
        } else {
          PackageContent::create($content);
        }
      }
    }

    Session::flash('success', __('Updated Successfully'));
    return 'success';
  }

  public function deletePackage(Request $request)
  {
    $package = Package::findOrFail($request->package_id);
    if ($package->packageLocationList()->count() > 0) {
      Session::flash('warning', __('First delete all the locations of this package!'));
      return redirect()->back();
    }

    if ($package->packagePlanList()->count() > 0) {
      Session::flash('warning', __('First delete all the plans of this package!'));
      return redirect()->back();
    }

    $pacakageBookings =  PackageBooking::where('package_id', $package->id)->where('user_id', Auth::guard('web')->user()->id)->get();
    //delete package bookings
    if ($pacakageBookings->count() > 0) {
      foreach ($pacakageBookings as $pb) {
        // first, delete the attachment
        Uploader::remove(public_path(Constant::WEBSITE_PACKAGE_BOOKING_ATTACHMENTS), $pb->attachment);
        // second, delete the invoice
        Uploader::remove(public_path(Constant::WEBSITE_PACKAGE_BOOKING_INVOICE), $pb->invoice);
        $pb->delete();
      }
    }

    // first, delete all the contents of this package
    $contents = $package->packageContent()->get();
    foreach ($contents as $content) {
      $content->delete();
    }

    // second, delete all the slider images of this package
    if (!is_null($package->slider_imgs)) {
      $images = json_decode($package->slider_imgs);
      foreach ($images as $image) {
        Uploader::remove(public_path(Constant::WEBSITE_PACKAGE_SLIDER_IMAGE),  $image);
      }
    }
    // third, delete featured image of this package
    Uploader::remove(public_path(Constant::WEBSITE_PACKAGE_IMAGE), $package->featured_img);
    $package->delete();
    Session::flash('success', __('Deleted Successfully'));
    return redirect()->back();
  }

  public function bulkDeletePackage(Request $request)
  {
    $ids = $request->ids;
    foreach ($ids as $id) {
      $package = Package::findOrFail($id);
      if ($package->packageLocationList()->count() > 0) {
        Session::flash('warning', __('First delete all the locations of those packages!'));
        return 'success';
      }
      if ($package->packagePlanList()->count() > 0) {
        Session::flash('warning', __('First delete all the plans of those packages!'));
        return 'success';
      }
      // first, delete all the contents of this package
      $contents = $package->packageContent()->get();
      foreach ($contents as $content) {
        $content->delete();
      }
      // second, delete all the slider images of this package
      if (!is_null($package->slider_imgs)) {
        $images = json_decode($package->slider_imgs);
        foreach ($images as $image) {
          Uploader::remove(public_path(Constant::WEBSITE_PACKAGE_SLIDER_IMAGE),  $image);
        }
      }
      // third, delete featured image of this package
      Uploader::remove(public_path(Constant::WEBSITE_PACKAGE_IMAGE), $package->featured_img);
      // finally, delete this package

      $pacakageBookings =  PackageBooking::where('package_id', $package->id)->where('user_id', Auth::guard('web')->user()->id)->get();
      //delete package bookings
      if ($pacakageBookings->count() > 0) {
        foreach ($pacakageBookings as $pb) {
          // first, delete the attachment
          Uploader::remove(public_path(Constant::WEBSITE_PACKAGE_BOOKING_ATTACHMENTS), $pb->attachment);
          Uploader::remove(public_path(Constant::WEBSITE_PACKAGE_BOOKING_INVOICE), $pb->invoice);
          $pb->delete();
        }
      }

      $package->delete();
    }

    Session::flash('success', __('Deleted Successfully'));
    return 'success';
  }

  public function storeLocation(Request $request)
  {
    $rule = [
      'name' => 'required',
      'latitude' => 'required | numeric| between:-90,90',
      'longitude' => 'required | numeric| between:-180,180',
    ];
    $validator = Validator::make($request->all(), $rule);
    if ($validator->fails()) {
      return Response::json([
        'errors' => $validator->getMessageBag()->toArray()
      ], 400);
    }
    $lang = Language::findOrFail($request->language_id);
    PackageLocation::create($request->except('language_id') + [
      'language_id' => $lang->id,
      'user_id' => Auth::guard('web')->user()->id,
    ]);

    Session::flash('success', __('Created Successfully'));
    return 'success';
  }

  public function viewLocations(Request $request, $package_id)
  {
    // first, get the language info from db
    $information['langs'] = Language::where('user_id', Auth::guard('web')->user()->id)->get();
    $language = Language::where('code', $request->language)->where('user_id', Auth::guard('web')->user()->id)->firstOrFail();
    $information['language'] = $language;
    $package = Package::findorFail($package_id);
    // then, get the locations of selected package
    $information['locations'] = PackageLocation::where('language_id', $language->id)
      ->where('package_id', $package_id)
      ->orderBy('id', 'desc')
      ->get();

    return view('user.packages.locations', $information);
  }

  public function updateLocation(Request $request)
  {
    $rule = [
      'name' => 'required'
    ];
    $validator = Validator::make($request->all(), $rule);
    if ($validator->fails()) {
      return Response::json([
        'errors' => $validator->getMessageBag()->toArray()
      ], 400);
    }

    PackageLocation::findOrFail($request->location_id)->update($request->all());
    Session::flash('success', __('Updated Successfully'));
    return 'success';
  }

  public function deleteLocation(Request $request)
  {
    PackageLocation::findOrFail($request->location_id)->delete();
    Session::flash('success', __('Deleted Successfully'));
    return redirect()->back();
  }

  public function bulkDeleteLocation(Request $request)
  {
    $ids = $request->ids;
    foreach ($ids as $id) {
      PackageLocation::findOrFail($id)->delete();
    }
    Session::flash('success', __('Deleted Successfully'));
    return 'success';
  }


  public function storeDaywisePlan(Request $request)
  {
    $rules = [
      'day_number' => 'required',
      'plan' => 'required'
    ];
    $validator = Validator::make($request->all(), $rules);
    if ($validator->fails()) {
      return Response::json([
        'errors' => $validator->getMessageBag()->toArray()
      ], 400);
    }

    $lang = Language::findOrFail($request->language_id);
    PackagePlan::create($request->except('language_id') + [
      'language_id' => $lang->id,
      'user_id' => Auth::guard('web')->user()->id,
    ]);
    Session::flash('success', __('Created Successfully'));
    return Response::json('success', 200);
  }

  public function storeTimewisePlan(Request $request)
  {
    $rules = [
      'start_time' => 'required',
      'end_time' => 'required',
      'plan' => 'required'
    ];
    $validator = Validator::make($request->all(), $rules);
    if ($validator->fails()) {
      return Response::json([
        'errors' => $validator->getMessageBag()->toArray()
      ], 400);
    }
    $lang = Language::findOrFail($request->language_id);
    PackagePlan::create($request->except('language_id') + [
      'language_id' => $lang->id,
      'user_id' => Auth::guard('web')->user()->id,
    ]);

    Session::flash('success', __('Created Successfully'));
    return Response::json('success', 200);
  }

  public function viewPlans(Request $request, $package_id)
  {
    $information['langs'] = Language::where('user_id', Auth::guard('web')->user()->id)->get();
    // first, get the language info from db
    $language = Language::where('code', $request->language)->where('user_id', Auth::guard('web')->user()->id)->firstOrFail();
    $information['language'] = $language;

    // then, get the plans of selected package
    $information['plans'] = PackagePlan::where('language_id', $language->id)
      ->where('package_id', $package_id)
      ->orderBy('id', 'desc')
      ->paginate(10);

    $package = Package::findOrFail($package_id);
    $information['package'] = $package;

    if ($package->plan_type == 'daywise') {
      return view('user.packages.daywise_plans', $information);
    } else if ($package->plan_type == 'timewise') {
      return view('user.packages.timewise_plans', $information);
    }
  }

  public function updateDaywisePlan(Request $request)
  {
    $rules = [
      'day_number' => 'required',
      'plan' => 'required'
    ];
    $validator = Validator::make($request->all(), $rules);
    if ($validator->fails()) {
      return Response::json([
        'errors' => $validator->getMessageBag()->toArray()
      ], 400);
    }
    PackagePlan::findOrFail($request->plan_id)->update($request->all());

    Session::flash('success', __('Updated Successfully'));
    return 'success';
  }

  public function updateTimewisePlan(Request $request)
  {
    $rules = [
      'start_time' => 'required',
      'end_time' => 'required',
      'plan' => 'required'
    ];
    $validator = Validator::make($request->all(), $rules);
    if ($validator->fails()) {
      return Response::json([
        'errors' => $validator->getMessageBag()->toArray()
      ], 400);
    }
    PackagePlan::findOrFail($request->plan_id)->update($request->all());

    Session::flash('success', __('Updated Successfully'));
    return 'success';
  }

  public function deletePlan(Request $request)
  {
    PackagePlan::findOrFail($request->plan_id)->delete();
    Session::flash('success', __('Updated Successfully'));
    return redirect()->back();
  }

  public function bulkDeletePlan(Request $request)
  {
    $ids = $request->ids;
    foreach ($ids as $id) {
      PackagePlan::findOrFail($id)->delete();
    }

    Session::flash('success', __('Deleted Successfully'));
    return 'success';
  }

  public function sliderImgageUpload(Request $request)
  {
    $img = $request->file('slider_images');
    $allowedExts = array('jpg', 'png', 'jpeg');
    $rules = [
      'slider_images' => [
        function ($attribute, $value, $fail) use ($img, $allowedExts) {
          $ext = $img->getClientOriginalExtension();
          if (!in_array($ext, $allowedExts)) {
            return $fail("Only png, jpg, jpeg images are allowed");
          }
        }
      ]
    ];
    $validator = Validator::make($request->all(), $rules);
    if ($validator->fails()) {
      return response()->json(['error' => $validator->errors()], 422);
    }


    if ($request->hasFile('slider_images')) {
      $filename = Uploader::upload_picture(Constant::WEBSITE_PACKAGE_SLIDER_IMAGE, $request->file('slider_images'));
    }
    return response()->json(['status' => 'success', 'file_id' => $filename]);
  }
  public function removeImage(Request $request)
  {
    $img = $request['imageName'];
    try {
      Uploader::remove(public_path(Constant::WEBSITE_PACKAGE_SLIDER_IMAGE), $img);
      return Response::json(['success' => __('The image has been deleted.')], 200);
    } catch (\Exception $e) {
      return Response::json(['error' => __('Something went wrong!')], 400);
    }
  }

  public function detachImage(Request $request)
  {
    $id = $request['id'];
    $key = $request['key'];

    $package = Package::query()->find($id);
    $sliderImages = json_decode($package->slider_imgs, true);
    if (count($sliderImages) == 1) {
      return Response::json(['message' => 'Sorry, the last image cannot be delete.'], 400);
    } else {
      $image = $sliderImages[$key];
      Uploader::remove(public_path(Constant::WEBSITE_PACKAGE_SLIDER_IMAGE), $image);

      array_splice($sliderImages, $key, 1);
      $package->update([
        'slider_imgs' => json_encode($sliderImages)
      ]);
      return Response::json(['message' => __('Deleted Successfully')], 200);
    }
  }

  public function bookings(Request $request)
  {
    $booking_number = null;

    if ($request->filled('booking_no')) {
      $booking_number = $request['booking_no'];
    }

    if (URL::current() == Route::is('tenant.package_bookings.all_bookings')) {
      $bookings = PackageBooking::when($booking_number, function ($query, $booking_number) {
        return $query->where('booking_number', 'like', '%' . $booking_number . '%');
      })->where('user_id', Auth::guard('web')->user()->id)->orderBy('id', 'desc')
        ->paginate(10);
    } else if (URL::current() == Route::is('tenant.package_bookings.paid_bookings')) {
      $bookings = PackageBooking::when($booking_number, function ($query, $booking_number) {
        return $query->where('booking_number', 'like', '%' . $booking_number . '%');
      })->where('user_id', Auth::guard('web')->user()->id)->where('payment_status', 1)
        ->orderBy('id', 'desc')
        ->paginate(10);
    } else if (URL::current() == Route::is('tenant.package_bookings.unpaid_bookings')) {
      $bookings = PackageBooking::when($booking_number, function ($query, $booking_number) {
        return $query->where('booking_number', 'like', '%' . $booking_number . '%');
      })->where('payment_status', 0)->where('user_id', Auth::guard('web')->user()->id)
        ->orderBy('id', 'desc')
        ->paginate(10);
    }

    return view('user.packages.bookings', compact('bookings'));
  }

  public function updatePaymentStatus(Request $request)
  {
    $packageBooking = PackageBooking::findOrFail($request->booking_id);

    $currentLang = session()->has('lang') ?
      (AdminLanguage::where('code', session()->get('lang'))->first())
      : (AdminLanguage::where('is_default', 1)->first());
    $bs = $currentLang->basic_setting;

    if ($request->payment_status == 1) {
      $packageBooking->update(['payment_status' => 1, 'payment_offline_status' => null]);
    } else {
      $packageBooking->update(['payment_status' => 0, 'payment_offline_status' => null]);
    }

    // delete previous invoice from local storage
    Uploader::remove(public_path(Constant::WEBSITE_PACKAGE_BOOKING_INVOICE), $packageBooking->invoice);
    // then, generate an invoice in pdf format
    $invoice = $this->generateInvoice($packageBooking);

    // update the invoice field information in database
    $packageBooking->update(['invoice' => $invoice]);
    // finally, send a mail to the customer with the invoice
    $this->sendMailForPaymentStatus($packageBooking, $request->payment_status);

    Session::flash('success',  __('Updated Successfully'));
    return redirect()->back();
  }

  public function bookingDetails($id)
  {

    $details = PackageBooking::findOrFail($id);
    $language = Language::where('user_id', Auth::guard('web')->user()->id)->firstOrFail();

    /**
     * to get the package title first get the package info using eloquent relationship
     * then, get the package content info of that package using eloquent relationship
     * after that, we can access the package title
     * also, get the package category using eloquent relationship
     */
    $packageInfo = $details->tourPackage()->firstOrFail();
    $packageContentInfo = $packageInfo->packageContent()->where('language_id', $language->id)
      ->firstOrFail();
    $packageTitle = $packageContentInfo->title;

    // Handling package category
    $packageCategoryName = null;
    if ($packageInfo->packageCategory()) {
      $category = $packageInfo->packageCategory()->where('language_id', $language->id)->first();
      if (!is_null($category)) {
        $packageCategoryName =  $category->name;
      }
    }

    return view(
      'user.packages.booking_details',
      compact('details', 'packageTitle', 'packageCategoryName')
    );
  }

  public function sendMail(Request $request)
  {
    $rules = [
      'subject' => 'required',
      'message' => 'required',
    ];
    $messages = [
      'subject.required' => __('The email subject field is required.'),
      'message.required' => __('The email message field is required.')
    ];
    $validator = Validator::make($request->all(), $rules, $messages);
    if ($validator->fails()) {
      return Response::json([
        'errors' => $validator->getMessageBag()->toArray()
      ], 400);
    }
    $be = BasicExtended::first();
    if (empty($be->smtp_host) || empty($be->smtp_port) || empty($be->encryption) || empty($be->smtp_username) || empty($be->smtp_password)) {
      return back();
    }

    if ($be->is_smtp == 1) {
      try {
        $smtp = [
          'transport' => 'smtp',
          'host' => $be->smtp_host,
          'port' => $be->smtp_port,
          'encryption' => $be->encryption,
          'username' => $be->smtp_username,
          'password' => $be->smtp_password,
          'timeout' => null,
          'auth_mode' => null,
        ];
        Config::set('mail.mailers.smtp', $smtp);
      } catch (\Exception $e) {
        Session::flash('error', $e->getMessage());
        return back();
      }
    }

    try {
      Mail::send([], [], function ($message) use ($request, $be) {
        $fromMail = $be->from_mail;
        $fromName = $be->from_name;
        $message->to($request->customer_email)
          ->subject($request->subject)
          ->from($fromMail, $fromName)
          ->html($request->message, 'text/html');
      });
      Session::flash('success', __('Mail sent successfully'));
      return 'success';
    } catch (\Exception $e) {
      Session::flash('error', $e->getMessage());
      return back();
    }
  }

  public function deleteBooking(Request $request, $id)
  {
    $packageBooking = PackageBooking::findOrFail($id);

    // first, delete the attachment
    Uploader::remove(public_path(Constant::WEBSITE_PACKAGE_BOOKING_ATTACHMENTS), $packageBooking->attachment);
    // second, delete the invoice
    Uploader::remove(public_path(Constant::WEBSITE_PACKAGE_BOOKING_INVOICE), $packageBooking->invoice);
    // finally, delete the package booking record from db
    $packageBooking->delete();

    Session::flash('success', __('Deleted Successfully'));
    return redirect()->back();
  }

  public function bulkDeleteBooking(Request $request)
  {
    $ids = $request->ids;
    foreach ($ids as $id) {
      $packageBooking = PackageBooking::findOrFail($id);
      // first, delete the attachment
      Uploader::remove(public_path(Constant::WEBSITE_PACKAGE_BOOKING_ATTACHMENTS), $packageBooking->attachment);
      // second, delete the invoice
      Uploader::remove(public_path(Constant::WEBSITE_PACKAGE_BOOKING_INVOICE), $packageBooking->invoice);

      $packageBooking->delete();
    }

    Session::flash('success', __('Deleted Successfully'));
    return 'success';
  }


  private function generateInvoice($bookingInfo)
  {
    $user = Auth::guard('web')->user();
    $fileName = $bookingInfo->booking_number . '.pdf';
    $directory = public_path(Constant::WEBSITE_PACKAGE_BOOKING_INVOICE . '/');
    if (!file_exists($directory)) {
      mkdir($directory, 0777, true);
    }
    $language = Language::query()->where('is_default', '=', 1)->where('user_id', '=', $user->id)->first();
    $fileLocated = $directory . $fileName;

    $keywords = get_keywords($user->id);

    Pdf::setOptions([
      'isHtml5ParserEnabled' => true,
      'isRemoteEnabled' => true,
      'logOutputFile' => storage_path('logs/log.htm'),
      'tempDir' => storage_path('logs/'),
    ])->loadView('user.pdf.package_booking', compact('bookingInfo', 'language', 'user', 'keywords'))->save($fileLocated);



    return $fileName;
  }

  private function sendMailForPaymentStatus($packageBooking, $status)
  {
    $user = Auth::guard('web')->user();

    if ($status == 1) {
      $mailTemplate = MailTemplate::where('mail_type', 'package_booking_payment_received')->where('user_id', Auth::guard('web')->user()->id)->firstOrFail();
    } else {
      $mailTemplate = MailTemplate::where('mail_type', 'package_booking_payment_cancelled')->where('user_id', Auth::guard('web')->user()->id)->firstOrFail();
    }
    $mailTemplate = str_replace(' ', '_', $mailTemplate->mail_type);

    $infoCus = DB::table('user_basic_settings')->where('user_id', '=', Auth::guard('web')->user()->id)->first();

    $mailer = new MegaMailer;
    $data = [
      'toMail' => $packageBooking->customer_email,
      'customer_name' => $packageBooking->customer_name,
      'website_title' => $infoCus->website_title,
      'templateType' => $mailTemplate,
      'type' => $mailTemplate
    ];
    $mailer->mailFromUser($data, $user);
    return 'success';
  }

  public function report(Request $request)
  {
    $queryResult['onlineGateways'] = OnlineGateway::query()->where('status', '=', 1)->where('user_id', Auth::guard('web')->user()->id)->get();
    $queryResult['offlineGateways'] = OfflineGateway::query()->where('status', '=', 1)->where('user_id', Auth::guard('web')->user()->id)->orderBy('serial_number', 'asc')->get();

    $from = $to = $paymentGateway = $paymentStatus = null;
    if ($request->filled('payment_gateway')) {
      $paymentGateway = $request->payment_gateway;
    }
    if ($request->filled('payment_status')) {
      $paymentStatus = $request->payment_status;
    }


    if ($request->filled('from') && $request->filled('to')) {
      $from = Carbon::parse($request->from)->toDateString();
      $to = Carbon::parse($request->to)->toDateString();
      $records = PackageBooking::query()
        ->where('user_id', Auth::guard('web')->user()->id)
        ->whereDate('created_at', '>=', $from)
        ->whereDate('created_at', '<=', $to)
        ->when($paymentGateway, function (Builder $query, $paymentGateway) {
          return $query->where('payment_method', '=', $paymentGateway);
        })
        ->when($paymentStatus, function (Builder $query, $paymentStatus) {
          $status =  $paymentStatus == 'incomplete' ? 0 : 1;
          return $query->where('payment_status', '=', $status);
        })
        ->select('*')->orderByDesc('id');
      $collection_1 = $this->manipulateCollection($records->get());
      Session::put('package_bookings', $collection_1);
      $collection_2 = $this->manipulateCollection($records->paginate(10));
      $queryResult['bookings'] = $collection_2;
    } else {
      Session::put('package_bookings', null);
      $queryResult['bookings'] = [];
    }


    return view('user.packages.report', $queryResult);
  }

  public function manipulateCollection($bookings)
  {
    $bookings->map(function ($booking) {
      //package title
      $package = $booking->tourPackage()->where('user_id', Auth::guard('web')->user()->id)->first();
      $booking['packageTitle'] = $package->packageContent()->where('package_id', $package->id)->pluck('title')->first();
      // format booking start date
      $startDateObj = Carbon::parse($booking->start_date);
      $booking['startDate'] = $startDateObj->format('M d, Y');

      // format booking end date
      $endDateObj = Carbon::parse($booking->end_date);
      $booking['endDate'] = $endDateObj->format('M d, Y');

      // format booking create date
      $createDateObj = Carbon::parse($booking->created_at);
      $booking['createdAt'] = $createDateObj->format('M d, Y');
    });

    return $bookings;
  }

  public function exportReport()
  {
    if (Session::has('package_bookings')) {
      $packageBooking = Session::get('package_bookings');

      if (count($packageBooking) == 0) {

        Session::flash('warning', __('No booking found to export!'));

        return redirect()->back();
      } else {
        return Excel::download(new PackageBookingsExport($packageBooking), 'package-bookings.csv');
      }
    } else {
      Session::flash('error', __('There has no booking to export.'));

      return redirect()->back();
    }
  }

  public function taxFee(Request $request)
  {
    $data['data'] = BasicSetting::select('package_tax_status', 'package_tax', 'package_fee_status', 'package_fee', 'package_tax', 'package_tax_status', 'package_fee_status', 'package_fee', 'base_currency_symbol')->where('user_id', Auth::guard('web')->user()->id)->first();

    return view('user.packages.tex_fee', $data);
  }
  public function updateTaxFee(Request $request)
  {
    $rules = [
      'package_tax_status' => 'required',
      'package_tax' => 'required_if:package_tax_status,1',
      'package_fee_status' => 'required',
      'package_fee' => 'required_if:room_fee_status,1',
    ];

    $validator = Validator::make($request->all(), $rules);
    if ($validator->fails()) {
      return response()->json(['errors' => $validator->errors()], 422);
    }
    BasicSetting::query()->updateOrInsert(
      ['user_id' => Auth::guard('web')->user()->id],
      $request->except(['_token', 'user_id'] + [
        'user_id' => Auth::guard('web')->user()->id,
      ])
    );
    Session::flash('success', __('Updated Successfully'));
    return "success";
  }
}
