<?php

namespace App\Http\Controllers\User\Journal;

use App\Http\Controllers\Controller;
use App\Models\User\Journal\BlogCategory;
use App\Models\User\Language;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Response;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Validator;

class CategoryController extends Controller
{
    public function index(Request $request)
    {
        $information['langs'] = Language::query()->where('user_id', Auth::guard('web')->user()->id)->get();
        $information['language'] = $information['langs']->where('code', $request->language)->first();
        if (is_null($information['language'])) {
            return view('errors.user-404');
        }
        $information['categories'] = BlogCategory::where('language_id', $information['language']->id)->where('user_id', Auth::guard('web')->user()->id)->orderByDesc('id')->get();

        return view('user.journal.category.index', $information);
    }

    public function store(Request $request, $language)
    {
        $rules = [
            'status' => 'required',
            'serial_number' => 'required|integer|min:0',
        ];

        $messages = [];
        $languages = Language::where('user_id', Auth::guard('web')->user()->id)->get();
        foreach ($languages as $lang) {
            $slug = slug_create($request[$lang->code . '_name']);
            $rules[$lang->code . '_name'] =
                [
                    'required',
                    'max:255',
                    function ($attribute, $value, $fail) use ($slug, $lang) {
                        $bis = BlogCategory::where('language_id', $lang->id)
                            ->where('user_id', Auth::guard('web')->user()->id)
                            ->get();
                        foreach ($bis as $key => $bi) {
                            if (strtolower($slug) == strtolower($bi->slug)) {
                                $fail('The name field must be unique for ' . $lang->name . ' language.');
                            }
                        }
                    }
                ];

            $messages[$lang->code . '_name.required'] = __('The category name field is required for') . ' ' . $lang->name . ' ' . __('language');
        }

        $validator = Validator::make($request->all(), $rules, $messages);
        if ($validator->fails()) {
            return Response::json([
                'errors' => $validator->getMessageBag()->toArray()
            ], 400);
        }

        $index_id = uniqid();
        foreach ($languages as $language) {
            $bcategory = new BlogCategory();
            $bcategory->indx = $index_id;
            $bcategory->language_id = $language->id;
            $bcategory->name = $request[$language->code . '_name'];
            $bcategory->slug = slug_create($request[$language->code . '_name']);
            $bcategory->status = $request->status;
            $bcategory->user_id = Auth::guard('web')->user()->id;
            $bcategory->serial_number = $request->serial_number;
            $bcategory->save();
        }
        Session::flash('success', __('Created Successfully'));
        return "success";
    }

    public function edit($id)
    {
        $user_id = Auth::guard('web')->user()->id;
        $data['languages'] = Language::where('user_id', $user_id)->get();
        $data['data'] = BlogCategory::findOrFail($id);
        return view('user.journal.category.edit', $data);
    }


    public function update(Request $request)
    {

        $rules = [
            'status' => 'required',
            'serial_number' => 'required|integer',
        ];

        $messages = [];
        $languages = Language::where('user_id', Auth::guard('web')->user()->id)->get();

        foreach ($languages as $language) {
            $slug = slug_create($request[$language->code . '_name']);
            $indx = $request->category_indx;
            $rules[$language->code . '_name'] = [
                'required',
                'max:255',
                function ($attribute, $value, $fail) use ($slug, $indx, $language) {
                    $bis = BlogCategory::where('indx', '<>', $indx)
                        ->where('language_id', $language->id)
                        ->where('user_id', Auth::guard('web')->user()->id)
                        ->get();
                    foreach ($bis as $key => $bi) {
                        if (strtolower($slug) == strtolower($bi->slug)) {
                            $fail(__('The title field must be unique for') . ' ' . $language->name . ' ' . __('language.'));
                        }
                    }
                }
            ];
            $messages[$language->code . '_name.required'] = __('The name field is required for') . ' ' . $language->name . ' ' . __('language');
        }
        $validator = Validator::make($request->all(), $rules, $messages);
        if ($validator->fails()) {
            return Response::json([
                'errors' => $validator->getMessageBag()->toArray()
            ], 400);
        }

        $user_id = Auth::guard('web')->user()->id;
        $languages = Language::where('user_id', $user_id)->get();
        $bcategory = BlogCategory::findOrFail($request->category_id);
        $indx = is_null($bcategory->indx) ? uniqid() : $bcategory->indx;

        foreach ($languages as $language) {
            $bcategory = BlogCategory::where('id', $request[$language->code . '_id'])->first();
            if (empty($bcategory)) {
                $bcategory = new BlogCategory();
            }
            $bcategory->indx = $indx;
            $bcategory->user_id = $user_id;
            $bcategory->language_id = $language->id;
            $bcategory->name = $request[$language->code . '_name'];
            $bcategory->slug = slug_create($request[$language->code . '_name']);
            $bcategory->status = $request->status;
            $bcategory->serial_number = $request->serial_number;
            $bcategory->save();
        }
        Session::flash('success', __('Updated Successfully'));
        return "success";
    }

    public function destroy($id)
    {
        $category = BlogCategory::query()->where('user_id', Auth::guard('web')->user()->id)->find($id);

        if ($category->blogContentList()->count() > 0) {
            Session::flash('warning', __('First delete all the blog related to this category!'));
            return redirect()->back();
        }
        $category->delete();
        return redirect()->back()->with('success', __('Deleted Successfully'));
    }

    public function bulkDestroy(Request $request)
    {
        $ids = $request->ids;
        foreach ($ids as $id) {
            $category = BlogCategory::query()->where('user_id', Auth::guard('web')->user()->id)->find($id);
            if ($category->blogContentList()->count() > 0) {
                Session::flash('warning', __('First delete all the blog related to this category!'));
                return 'success';
            }
            $category->delete();
        }
        Session::flash('success', __('Deleted Successfully'));
        return "success";
    }
}
