<?php

namespace App\Http\Controllers\User\Journal;

use App\Constants\Constant;
use App\Http\Controllers\Controller;
use App\Http\Helpers\Uploader;
use App\Http\Requests\Blog\StoreRequest;
use App\Http\Requests\Blog\UpdateRequest;
use App\Models\User\Journal\Blog;
use App\Models\User\Journal\BlogInformation;
use App\Models\User\Language;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Session;
use Mews\Purifier\Facades\Purifier;

class BlogController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return
     */
    public function index(Request $request)
    {
        $information['langs'] = Language::query()->where('user_id', Auth::guard('web')->user()->id)->get();
        $information['language'] = $information['langs']->where('code', $request->language)->first();
        if (is_null($information['language'])) {
            return view('errors.user-404');
        }
        $information['blogs'] = DB::table('user_blogs')
            ->join('user_blog_categories', 'user_blogs.category_index', '=', 'user_blog_categories.indx')
            ->join('user_blog_informations', 'user_blogs.id', '=', 'user_blog_informations.blog_id')
            ->where('user_blog_informations.language_id', '=', $information['language']->id)
            ->where('user_blog_categories.language_id', '=', $information['language']->id)
            ->where('user_blog_informations.user_id', '=', Auth::guard('web')->user()->id)
            ->select('user_blogs.id', 'user_blogs.serial_number', 'user_blogs.created_at', 'user_blog_informations.title', 'user_blog_informations.author', 'user_blogs.image', 'user_blog_categories.name as categoryName')
            ->orderByDesc('user_blogs.id')
            ->get();
        return view('user.journal.blog.index', $information);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return
     */
    public function create()
    {
        // get all the languages from db
        $information['languages'] = Language::query()->where('user_id', Auth::guard('web')->user()->id)->get();
        $information['defaultLang'] = $information['languages']->where('is_default', 1)->first();
        return view('user.journal.blog.create', $information);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param Request $request
     * @return
     */
    public function store(StoreRequest $request, $language)
    {
        // store image in storage
        $language = Language::query()->where('code', $language)->first();
        $imgName = Uploader::upload_picture(public_path(Constant::WEBSITE_BLOG_IMAGE), $request->file('image'));

        // store data in db
        $blog = Blog::create($request->except('image', 'user_id') + [
            'image' => $imgName,
            'user_id' => Auth::guard('web')->user()->id,
            'language_id' => $language->id,
            'category_index' => $request->category_id,
        ]);
        
        $languages = Language::query()->where('user_id', Auth::guard('web')->user()->id)->get();

        foreach ($languages as $language) {
            if ($request->input($language->code . '_title') || $request->input($language->code . '_author') || $request->input($language->code . '_content') || $request->input($language->code . '_meta_keywords') || $request->input($language->code . '_meta_description')) {
                $blogInformation = new BlogInformation();
                $blogInformation->language_id = $language->id;
                $blogInformation->user_id = Auth::guard('web')->user()->id;
                $blogInformation->blog_id = $blog->id;
                $blogInformation->title = $request[$language->code . '_title'];
                $blogInformation->slug = make_slug($request[$language->code . '_title']);
                $blogInformation->author = $request[$language->code . '_author'];
                $blogInformation->content = Purifier::clean($request[$language->code . '_content'], 'youtube');
                $blogInformation->meta_keywords = $request[$language->code . '_meta_keywords'];
                $blogInformation->meta_description = $request[$language->code . '_meta_description'];
                $blogInformation->save();
            }
        }

        Session::flash('success', __('Created Successfully'));
        return "success";
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param int $id
     * @return
     */
    public function edit($id)
    {
        $information['blog'] = Blog::query()->where('user_id', Auth::guard('web')->user()->id)->findorFail($id);  // get all the languages from db
        $information['languages'] = Language::query()->where('user_id', Auth::guard('web')->user()->id)->get();
        $information['defaultLang'] = $information['languages']->where('is_default', 1)->first();
        return view('user.journal.blog.edit', $information);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param Request $request
     * @param int $id
     * @return
     */
    public function update(UpdateRequest $request, $id)
    {

        $blog = Blog::query()->where('user_id', Auth::guard('web')->user()->id)->find($id);
        // store new image in storage
        if ($request->hasFile('image')) {
            $imgName = Uploader::update_picture(public_path(Constant::WEBSITE_BLOG_IMAGE), $request->file('image'), $blog->image);
        }
        // update data in db
        $blog->update($request->except('image', 'category_id') + [
            'image' => $request->hasFile('image') ? $imgName : $blog->image,
            'category_index' => $request->category_id
        ]);
        $languages = Language::where('user_id', Auth::guard('web')->user()->id)->get();

        foreach ($languages as $language) {
            $hasExistingContent = BlogInformation::where('blog_id', $id)
                ->where('language_id', $language->id)
                ->exists();
            if (
                $hasExistingContent ||
                $request[$language->code . '_title'] ||
                $request[$language->code . '_author'] ||
                $request[$language->code . '_content'] ||
                $request[$language->code . '_meta_keywords'] ||
                $request[$language->code . '_meta_description']
            ) {

                BlogInformation::query()->updateOrCreate([
                    'blog_id' => $id,
                    'user_id' => Auth::guard('web')->user()->id,
                    'language_id' => $language->id
                ], [
                    'title' => $request[$language->code . '_title'],
                    'slug' => make_slug($request[$language->code . '_title']),
                    'author' => $request[$language->code . '_author'],
                    'content' => Purifier::clean($request[$language->code . '_content'], 'youtube'),
                    'user_id' => Auth::guard('web')->user()->id,
                    'language_id' => $language->id,
                    'meta_keywords' => $request[$language->code . '_meta_keywords'],
                    'meta_description' => $request[$language->code . '_meta_description']
                ]);
            }
        }

        Session::flash('success', __('Updated Successfully'));
        return "success";
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param int $id
     * @return
     */
    public function destroy($id)
    {
        $blog = Blog::query()->where('user_id', Auth::guard('web')->user()->id)->find($id);
        // first, delete the image
        Uploader::remove(public_path(Constant::WEBSITE_BLOG_IMAGE), $blog->image);
        $blogInformations = BlogInformation::query()->where('blog_id', $blog->id)->where('user_id', Auth::guard('web')->user()->id)->get();
        foreach ($blogInformations as $blogInformation) {
            $blogInformation->delete();
        }
        $blog->delete();
        return redirect()->back()->with('success', __('Deleted Successfully'));
    }

    /**
     * Remove the selected or all resources from storage.
     *
     * @param Request $request
     * @return
     */
    public function bulkDestroy(Request $request)
    {
        $ids = $request->ids;
        foreach ($ids as $id) {
            $blog = Blog::query()->where('user_id', Auth::guard('web')->user()->id)->find($id);
            // first, delete the image
            Uploader::remove(public_path(Constant::WEBSITE_BLOG_IMAGE), $blog->image);
            $blogInformations = BlogInformation::query()->where('blog_id', $blog->id)->where('user_id', Auth::guard('web')->user()->id)->get();
            foreach ($blogInformations as $blogInformation) {
                $blogInformation->delete();
            }
            $blog->delete();
        }

        Session::flash('success', __('Deleted Successfully'));
        return "success";
    }
}
