<?php

namespace App\Http\Controllers\User\HomePage;

use App\Constants\Constant;
use App\Http\Controllers\Controller;
use App\Http\Helpers\Uploader;
use App\Models\User\HomePage\Brand;
use App\Models\User\Language;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Response;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Validator;

class BrandSectionController extends Controller
{
  public function brandSection(Request $request)
  {
    // first, get the language info from db
    $language = Language::where('code', $request->language)->where('user_id', Auth::guard('web')->user()->id)->firstorFail();
    $information['language'] = $language;

    // also, get the brand info of that language from db
    $information['brands'] = Brand::where('language_id', $language->id)->where('user_id', Auth::guard('web')->user()->id)
      ->orderBy('id', 'desc')
      ->get();

    // get all the languages from db
    $information['langs'] = Language::where('user_id', Auth::guard('web')->user()->id)->get();

    return view('user.home_page.brand_section.index', $information);
  }

  public function storeBrand(Request $request, $language)
  {
    $rules = [
      'brand_url' => 'required',
      'serial_number' => 'required',
      'brand_img' => 'required|image|mimes:jpeg,png,jpg|max:2048'
    ];

    $validator = Validator::make($request->all(), $rules);
    if ($validator->fails()) {
      return Response::json([
        'errors' => $validator->getMessageBag()->toArray()
      ], 400);
    }

    $lang = Language::where('code', $language)->where('user_id', Auth::guard('web')->user()->id)->first();

    // set a name for the image and store it to local storage
    $brandImgName = Uploader::upload_picture(public_path(Constant::WEBSITE_BRAND_IMAGE), $request->file('brand_img'));

    Brand::create($request->except('language_id', 'brand_img') + [
      'language_id' => $lang->id,
      'user_id' => Auth::guard('web')->user()->id,
      'brand_img' => $brandImgName
    ]);

    Session::flash('success', __('Created Successfully'));
    return 'success';
  }

  public function updateBrand(Request $request)
  {
    $rules = [
      'brand_url' => 'required',
      'serial_number' => 'required'
    ];
    $validator = Validator::make($request->all(), $rules);
    if ($validator->fails()) {
      return Response::json([
        'errors' => $validator->getMessageBag()->toArray()
      ], 400);
    }

    $brand = Brand::where('id', $request->brand_id)->where('user_id', Auth::guard('web')->user()->id)->first();

    if ($request->hasFile('brand_img')) {
      // set a name for the image and store it to local storage
      $brandImgName = Uploader::update_picture(public_path(Constant::WEBSITE_BRAND_IMAGE), $request->file('brand_img'), $brand->brand_img);
    }
    $brand->update($request->except('brand_img') + [
      'brand_img' => $request->hasFile('brand_img') ? $brandImgName : $brand->brand_img
    ]);
    Session::flash('success', __('Updated Successfully'));
    return 'success';
  }

  public function deleteBrand(Request $request)
  {
    $brand = Brand::where('id', $request->brand_id)->first();
    if (!is_null($brand->brand_img) && file_exists(public_path(Constant::WEBSITE_BRAND_IMAGE .'/' . $brand->brand_img))) {
      Uploader::remove(public_path(Constant::WEBSITE_BRAND_IMAGE), $brand->brand_img);
    }
    $brand->delete();
    
    Session::flash('success', __('Deleted Successfully'));
    return redirect()->back();
  }
}
