<?php

namespace App\Http\Controllers\User;

use App\Constants\Constant;
use App\Http\Controllers\Controller;
use App\Http\Helpers\Uploader;
use App\Models\User\Gallery;
use App\Models\User\GalleryCategory;
use App\Models\User\Language;
use App\Rules\ImageMimeTypeRule;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Response;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Validator;

class GalleryController extends Controller
{
  public function categories(Request $request)
  {
    // first, get the language info from db
    $language = Language::where('code', $request->language)->where('user_id', Auth::guard('web')->user()->id)->firstOrFail();
    $information['language'] = $language;
    // then, get the gallery categories of that language from db
    $information['galleryCategories'] = GalleryCategory::where('language_id', $language->id)
      ->orderBy('id', 'desc')
      ->paginate(10);
    // also, get all the languages from db
    $information['langs'] = Language::where('user_id', Auth::guard('web')->user()->id)->get();
    return view('user.gallery.categories', $information);
  }

  public function storeCategory(Request $request, $language)
  {
    $rules = [
      'name' => 'required',
      'status' => 'required',
      'serial_number' => 'required'
    ];
    $validator = Validator::make($request->all(), $rules);
    if ($validator->fails()) {
      return Response::json([
        'errors' => $validator->getMessageBag()->toArray()
      ], 400);
    }
    $lang = Language::where('code', $language)->where('user_id', Auth::guard('web')->user()->id)->firstOrFail();
    GalleryCategory::create($request->except('language_id', '_token') + [
      'language_id' => $lang->id,
      'user_id' => Auth::guard('web')->user()->id
    ]);

    Session::flash('success', __('Created Successfully'));
    return 'success';
  }

  public function updateCategory(Request $request)
  {
    $rules = [
      'name' => 'required',
      'status' => 'required',
      'serial_number' => 'required'
    ];
    $validator = Validator::make($request->all(), $rules);

    if ($validator->fails()) {
      return Response::json([
        'errors' => $validator->getMessageBag()->toArray()
      ], 400);
    }
    GalleryCategory::findOrFail($request->category_id)->update($request->all());

    Session::flash('success', __('Updated Successfully'));
    return 'success';
  }

  public function deleteCategory(Request $request)
  {
    $galleryCategory = GalleryCategory::findOrFail($request->category_id);
    if ($galleryCategory->galleryImgList()->count() > 0) {
      Session::flash('warning', __('First delete all the images of this category!'));
      return redirect()->back();
    }
    $galleryCategory->delete();

    Session::flash('success', __('Deleted Successfully'));
    return redirect()->back();
  }

  public function bulkDeleteCategory(Request $request)
  {
    $ids = $request->ids;
    foreach ($ids as $id) {
      $galleryCategory = GalleryCategory::findOrFail($id);
      if ($galleryCategory->galleryImgList()->count() > 0) {
        Session::flash('warning', __('First delete all the images of those category!'));
        return 'success';
      }
      $galleryCategory->delete();
    }

    Session::flash('success', __('Deleted Successfully'));
    return 'success';
  }


  public function index(Request $request)
  {
    // first, get the language info from db
    $language = Language::where('code', $request->language)->where('user_id', Auth::guard('web')->user()->id)->firstOrFail();
    // get all the gallery categories of that language from db
    $information['categories'] = GalleryCategory::where('language_id', $language->id)->where('user_id', Auth::guard('web')->user()->id)
      ->where('status', 1)
      ->orderBy('serial_number', 'asc')
      ->get();
    // then, get the gallery images of that language from db
    $information['galleryInfos'] = Gallery::with('galleryCategory')->where('user_id', Auth::guard('web')->user()->id)
      ->where('language_id', $language->id)
      ->orderBy('id', 'desc')
      ->get();

    return view('user.gallery.index', $information);
  }

  public function storeInfo(Request $request, $language)
  {
    $rules = [
      'gallery_category_id' => 'required',
      'title' => 'required',
      'serial_number' => 'required',
      'gallery_img' => [
        'required',
        new ImageMimeTypeRule()
      ]
    ];
    $messages = [
      'gallery_category_id.required' => __('The category field is required'),
      'gallery_img.required' => __('The image field is required'),
    ];
    $validator = Validator::make($request->all(), $rules, $messages);
    if ($validator->fails()) {
      return Response::json([
        'errors' => $validator->getMessageBag()->toArray()
      ], 400);
    }
    $gallery_img = Uploader::upload_picture(public_path(Constant::WEBSITE_GALLERY_IMAGE), $request->file('gallery_img'));

    $language = Language::where('code', $language)->where('user_id', Auth::guard('web')->user()->id)->firstOrFail();
    // set a name for the image and store it to local storage
    Gallery::create($request->except('language_id', 'gallery_img') + [
      'language_id' => $language->id,
      'user_id' => Auth::guard('web')->user()->id,
      'gallery_img' => $gallery_img
    ]);

    Session::flash('success', __('Created Successfully'));
    return 'success';
  }

  public function updateInfo(Request $request)
  {
    $rules = [
      'gallery_category_id' => 'required',
      'title' => 'required',
      'serial_number' => 'required'
    ];

    $validator = Validator::make($request->all(), $rules);
    if ($validator->fails()) {
      return Response::json([
        'errors' => $validator->getMessageBag()->toArray()
      ], 400);
    }

    $info = Gallery::where('id', $request->gallery_id)->where('user_id', Auth::guard('web')->user()->id)->firstOrFail();

    if ($request->hasFile('gallery_img')) {
      Uploader::remove(public_path(Constant::WEBSITE_GALLERY_IMAGE),$info->gallery_img);
      $gallery_img = Uploader::upload_picture(public_path(Constant::WEBSITE_GALLERY_IMAGE), $request->file('gallery_img'));
    }

    $info->update($request->except('gallery_img') + [
      'gallery_img' => $request->hasFile('gallery_img') ? $gallery_img : $info->gallery_img
    ]);

    Session::flash('success', __('Updated Successfully'));
    return 'success';
  }

  public function deleteInfo(Request $request)
  {
    $info = Gallery::where('id', $request->gallery_id)->firstOrFail();
    Uploader::remove(public_path(Constant::WEBSITE_GALLERY_IMAGE), $info->gallery_img);
    $info->delete();

    Session::flash('success', __('Deleted Successfully'));
    return redirect()->back();
  }

  public function bulkDeleteInfo(Request $request)
  {
    $ids = $request->ids;
    foreach ($ids as $id) {
      $info = Gallery::findOrFail($id);
      Uploader::remove(public_path(Constant::WEBSITE_GALLERY_IMAGE), $info->gallery_img);
      $info->delete();
    }

    Session::flash('success', __('Deleted Successfully'));
    return 'success';
  }
}
