<?php

namespace App\Http\Controllers\User;

use App\Constants\Constant;
use App\Enums\ThemeVersion;
use App\Http\Controllers\Controller;
use App\Http\Helpers\Uploader;
use App\Models\Timezone;
use App\Models\User\BasicExtended;
use App\Models\User\BasicSetting;
use App\Models\User\HomePage\SectionHeading;
use App\Models\User\Language;
use App\Models\User\SEO;
use App\Rules\ImageMimeTypeRule;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rules\Enum;
use Mews\Purifier\Facades\Purifier;

class BasicController extends Controller
{


    public function breadcrumb(Request $request)
    {
        $data['basic_setting'] = BasicSetting::where('user_id', Auth::guard('web')->user()->id)->select('breadcrumb')->first();
        return view('user.settings.breadcrumb', $data);
    }

    public function updateBreadcrumb(Request $request)
    {
        $bss = BasicSetting::where('user_id', Auth::guard('web')->user()->id)->select('breadcrumb')->first();

        $rules = [];
        if (!$request->filled('breadcrumb') && is_null($bss->breadcrumb)) {
            $rules['breadcrumb'] = 'required';
        }
        if ($request->hasFile('breadcrumb')) {
            $rules['breadcrumb'] = new ImageMimeTypeRule();
        }

        $validator = Validator::make($request->all(), $rules);
        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors(), 'id' => 'breadcrumb'], 422);
        }

        if ($request->hasFile('breadcrumb')) {
            $filename = Uploader::update_picture(public_path(Constant::WEBSITE_BREADCRUMB), $request->file('breadcrumb'), $bss->breadcrumb);
            BasicSetting::query()->updateOrInsert(
                ['user_id' => Auth::guard('web')->user()->id],
                ['breadcrumb' => $filename]
            );
        }
        Session::flash('success', __('Updated Successfully'));
        return "success";
    }

    public function footerLogo(Request $request)
    {
        $data['basic_setting'] = BasicSetting::where('user_id', Auth::guard('web')->user()->id)->select('footer_logo')->first();
        return view('user.settings.footer-logo', $data);
    }

    public function updateFooterLogo(Request $request)
    {
        $bss = BasicSetting::where('user_id', Auth::guard('web')->user()->id)->select('footer_logo')->first();

        $rules = [];
        if (!$request->filled('footer_logo') && is_null($bss->footer_logo)) {
            $rules['footer_logo'] = 'required';
        }
        if ($request->hasFile('footer_logo')) {
            $rules['footer_logo'] = new ImageMimeTypeRule();
        }

        $validator = Validator::make($request->all(), $rules);
        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors(), 'id' => 'footer_logo'], 422);
        }

        if ($request->hasFile('footer_logo')) {
            $filename = Uploader::upload_picture(Constant::WEBSITE_FOOTER_LOGO, $request->file('footer_logo'));
            BasicSetting::query()->updateOrInsert(
                ['user_id' => Auth::guard('web')->user()->id],
                ['footer_logo' => $filename]
            );
        }
        Session::flash('success', __('Updated Successfully'));
        return "success";
    }


    public function seo(Request $request)
    {
        // first, get the language info from db
        $language = Language::where('code', $request->language)->where('user_id', Auth::user()->id)->firstOrFail();
        $langId = $language->id;
        // then, get the seo info of that language from db
        $seo = SEO::where('language_id', $langId)->where('user_id', Auth::user()->id);
        if ($seo->count() == 0) {
            // if seo info of that language does not exist then create a new one
            SEO::create($request->except('language_id', 'user_id') + [
                'language_id' => $langId,
                'user_id' => Auth::user()->id,
            ]);
        }

        $information['language'] = $language;
        // then, get the seo info of that language from db
        $information['data'] = $seo->first();
        // get all the languages from db
        $information['langs'] = Language::where('user_id', Auth::user()->id)->get();

        return view('user.settings.seo', $information);
    }

    public function updateSEO(Request $request)
    {
        // first, get the language info from db
        $language = Language::where('code', $request->userLanguage)->where('user_id', Auth::user()->id)->first();
        // then, get the seo info of that language from db
        SEO::query()->updateOrInsert(
            [
                'user_id' => Auth::guard('web')->user()->id,
                'language_id' => $language->id,
            ],
            $request->except(['_token', 'userLanguage']) + [
                'language_id' => $language->id,
                'user_id' => Auth::guard('web')->user()->id,
            ]
        );
        Session::flash('success', __('Updated Successfully'));
        return redirect()->back();
    }

    public function information()
    {
        $data = BasicSetting::query()->where('user_id', Auth::guard('web')->user()->id)
            ->select('website_title', 'support_email', 'support_contact', 'address', 'latitude', 'longitude')
            ->first();
        return view('user.settings.information', ['data' => $data]);
    }

    public function updateInfo(Request $request)
    {
        $request->validate(
            [
                'support_email' => 'nullable',
                'support_contact' => 'nullable',
                'address' => 'nullable',
                'latitude' => 'nullable|numeric',
                'longitude' => 'nullable|numeric',
            ]
        );
        BasicSetting::query()->updateOrInsert(
            ['user_id' => Auth::guard('web')->user()->id],
            $request->except(['_token', 'user_id'] + [
                'user_id' => Auth::guard('web')->user()->id,
            ])
        );
        Session::flash('success', __('Updated Successfully'));
        return redirect()->back();
    }

    public function plugins()
    {
        $data = BasicSetting::query()->where('user_id', Auth::guard('web')->user()->id)
            ->orderBy('id', 'desc')
            ->first();

        return view('user.settings.plugins', ['data' => $data]);
    }

    public function maintenance()
    {
        $data = BasicSetting::query()->where('user_id', Auth::guard('web')->user()->id)
            ->select('maintenance_img', 'maintenance_status', 'maintenance_msg', 'bypass_token')
            ->first();
        return view('user.settings.maintenance', ['data' => $data]);
    }

    public function updateMaintenance(Request $request)
    {

        $data = BasicSetting::query()
            ->where('user_id', Auth::guard('web')->user()->id)
            ->select('maintenance_img')
            ->first();
        $rules = $messages = [];
        if (!$request->filled('maintenance_img') && is_null($data->maintenance_img)) {
            $rules['maintenance_img'] = 'required';
            $messages['maintenance_img.required'] = __('The maintenance image field is required.');
        }
        if ($request->hasFile('maintenance_img')) {
            $rules['maintenance_img'] = new ImageMimeTypeRule();
        }

        $rules['maintenance_status'] = 'required';
        $rules['maintenance_msg'] = 'required';

        $messages['maintenance_msg.required'] = __('The maintenance message field is required.');
        $validator = Validator::make($request->all(), $rules);
        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors(), 'id' => 'footer_logo'], 422);
        }

        if ($request->hasFile('maintenance_img')) {
            $imageName = Uploader::update_picture(
                public_path(Constant::WEBSITE_MAINTENANCE_IMAGE),
                $request->file('maintenance_img'),
                $data->maintenance_img
            );
        }
        BasicSetting::query()->updateOrInsert(
            ['user_id' => Auth::guard('web')->user()->id],
            $request->except(['_token', 'user_id', 'maintenance_img', 'maintenance_msg']) + [
                'maintenance_img' => $request->hasFile('maintenance_img') ? $imageName : $data->maintenance_img,
                'maintenance_msg' => Purifier::clean($request->maintenance_msg, 'youtube'),
                'user_id' => Auth::guard('web')->user()->id,
            ]
        );

        Session::flash('success', __('Updated Successfully'));
        return "success";
    }

    public function getMailInformation()
    {
        $data['info'] = BasicSetting::where('user_id', Auth::guard('web')->user()->id)->select('reply_to', 'from_name')->first();
        return view('user.settings.email.mail-information', $data);
    }

    public function storeMailInformation(Request $request)
    {
        $request->validate([
            'email' => 'required',
            'from_name' => 'required',
        ], [
            'email.required' => __('The email field is required'),
            'from_name.required' => __('The from name field is required'),
        ]);
        $info = BasicSetting::where('user_id', Auth::guard('web')->user()->id)->first();
        $info->reply_to = $request->email;
        $info->from_name = $request->from_name;
        $info->save();
        Session::flash('success', __('Updated Successfully'));
        return back();
    }

    public function updateScript(Request $request)
    {
        $basicSettings = BasicSetting::where('user_id', Auth::guard('web')->user()->id)->first();
        $basicSettings->update([
            'google_recaptcha_status' => $request->google_recaptcha_status,
            'google_recaptcha_site_key' => $request->google_recaptcha_site_key,
            'google_recaptcha_secret_key' => $request->google_recaptcha_secret_key,
            'is_disqus' => $request->is_disqus,
            'disqus_shortname' => $request->disqus_shortname,
            'is_tawkto' => $request->is_tawkto,
            'tawkto_chat_link' => $request->tawkto_chat_link,
            'is_whatsapp' => $request->is_whatsapp,
            'whatsapp_number' => $request->whatsapp_number,
            'whatsapp_header_title' => $request->whatsapp_header_title,
            'whatsapp_popup_message' => Purifier::clean($request->whatsapp_popup_message),
            'whatsapp_popup' => $request->whatsapp_popup,
            'google_login_status' => $request->google_login_status,
            'google_client_id' => $request->google_client_id,
            'google_client_secret' => $request->google_client_secret,
        ]);

        Session::flash('success', __('Updated Successfully'));
        return redirect()->back();
    }
    public function generalSettings(Request $request)
    {
        $data['timezones'] = Timezone::all();
        $data['data'] = BasicSetting::query()
            ->where('user_id', Auth::guard('web')->user()->id)
            ->first();
        return view('user.settings.general_settings', $data);
    }

    public function updateGeneralSettings(Request $request)
    {
        $userBs = BasicSetting::query()
            ->where('user_id', Auth::guard('web')->user()->id)
            ->first();
        $rules = [
            'website_title' => 'required',
            'favicon' => [is_null($userBs->favicon) ? 'required' : 'nullable', new ImageMimeTypeRule()],
            'thumbnail_image' => [is_null($userBs->logo) ? 'required' : 'nullable'],
            'preloader' => [
                is_null($userBs->preloader) ? 'required' : 'nullable',
                new ImageMimeTypeRule()
            ],
            'primary_color' => 'required',
            'primary_color' => 'required',
            'secondary_color' => 'required',
            'breadcrumb_overlay_color' => 'required',
            'breadcrumb_overlay_opacity' => 'required|numeric|min:0|max:1',
            'base_currency_symbol' => 'required',
            'base_currency_symbol_position' => 'required',
            'base_currency_text' => 'required',
            'base_currency_text_position' => 'required',
            'base_currency_rate' => 'required|numeric',
            'timezone' => 'required',
            'theme_version' => ['required', new Enum(ThemeVersion::class)],
            'home_version' => 'required'

        ];
        $validator = Validator::make($request->all(), $rules);
        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        $favicon = $userBs->favicon;
        if ($request->hasFile('favicon')) {
            $favicon = Uploader::update_picture(
                public_path(Constant::WEBSITE_FAVICON),
                $request->file('favicon'),
                $userBs->favicon
            );
        }
        $preloader = $userBs->preloader;
        if ($request->hasFile('preloader')) {
            $preloader = Uploader::update_picture(
                public_path(Constant::WEBSITE_FAVICON),
                $request->file('preloader'),
                $userBs->preloader
            );
        }
        //thumnail image is logo image
        $logo = $userBs->logo;
        if ($request->hasFile('thumbnail_image')) {
            $weight = in_array($userBs->theme_version,['theme_three','theme_four','theme_five']) ? 200 : 322; 
            $height = in_array($userBs->theme_version,['theme_three','theme_four','theme_five']) ? 71 : 115; 
            $logo = Uploader::logo_image(public_path(Constant::WEBSITE_LOGO . '/'), $request->file('thumbnail_image'), $userBs->logo, $weight, $height);
        }
        BasicSetting::query()->updateOrInsert(
            ['user_id' => Auth::guard('web')->user()->id],
            [
                'website_title' => $request->website_title,
                'base_currency_symbol' => $request->base_currency_symbol,
                'base_currency_symbol_position' => $request->base_currency_symbol_position,
                'base_currency_text' => $request->base_currency_text,
                'base_currency_text_position' => $request->base_currency_text_position,
                'base_currency_rate' => $request->base_currency_rate,
                'timezone' => $request->timezone,
                'theme_version' => $request->theme_version,
                'primary_color' => $request->primary_color,
                'secondary_color' => $request->secondary_color,
                'breadcrumb_overlay_color' => $request->breadcrumb_overlay_color,
                'breadcrumb_overlay_opacity' => $request->breadcrumb_overlay_opacity,
                'breadcrumb_overlay_opacity' => $request->breadcrumb_overlay_opacity,
                'home_version' => $request->home_version,
                'favicon' => $favicon,
                'logo' => $logo,
                'preloader' => $preloader,
                'preloader_status' => $request->preloader_status,
            ]
        );
        Session::flash('success', __('Updated Successfully'));
        return "success";
    }
    public function images_texts(Request $request)
    {
        $language = Language::where('code', $request->language)->where('user_id', Auth::guard('web')->user()->id)->firstOrFail();
        $data['language'] = $language;
        $data['data'] = SectionHeading::firstOrCreate(
            ['user_id' => Auth::guard('web')->user()->id, 'language_id' => $language->id]
        );
        return view('user.settings.images_texts', $data);
    }

    public function updateImages_texts(Request $request)
    {
        $languageId = $request->language_id;
        $userBs = BasicSetting::query()
            ->where('user_id', Auth::guard('web')->user()->id)
            ->first();

        $secHeading = SectionHeading::query()
            ->where('user_id', Auth::guard('web')->user()->id)
            ->where('language_id', $languageId)
            ->first();
        $rules = [
            'room_section_title' => 'nullable|max:255',
            'room_category_section_title' => 'nullable|max:255',
            'room_section_subtitle' => 'nullable|max:255',
            'room_section_text' => 'nullable',
            'service_section_title' => 'nullable|max:255',
            'service_section_subtitle' => 'nullable|max:255',

            'video_section_title' => 'nullable|max:255',
            'video_section_subtitle' => 'nullable|max:255',
            'video_section_button' => 'nullable|max:255',
            'video_section_button_url' => 'nullable|max:255',
            'video_section_video_url' => 'nullable|max:255',

            'package_section_title' => 'nullable|max:255',
            'package_section_subtitle' => 'nullable|max:255',

            'facility_section_title' => 'nullable|max:255',
            'facility_section_subtitle' => 'nullable|max:255',

            'testimonial_section_title' => 'nullable|max:255',
            'testimonial_section_subtitle' => 'nullable|max:255',

            'intro_primary_title' => 'nullable|max:255',
            'intro_secondary_title' => 'nullable|max:255',
            'intro_text' => 'nullable',
            'intro_button_text' => 'nullable | max:255',
            'intro_button_url' => 'nullable |max:255',

            'hero_section_title' => 'nullable|max:255',
            'hero_section_subtitle' => 'nullable|max:255',
            'hero_section_button_name' => 'nullable|max:255',
            'hero_section_button_url' => 'nullable',

            'blog_section_title' => 'nullable|max:255',
            'blog_section_subtitle' => 'nullable|max:255',

            'faq_section_title' => 'nullable|max:255',
            'faq_section_subtitle' => 'nullable|max:255',
            'hero_section_video_link' => 'nullable',
            
            'latest_room_section_title' => 'nullable|max:255',
            'latest_package_section_title' => 'nullable|max:255',
            'intro_member_image_show' => 'nullable|max:255',
            'package_category_section_title' => 'nullable|max:255',
            
        ];

        //theme version 1
        if (in_array($userBs->theme_version, [ThemeVersion::ThemeOne->value])) {
            if(in_array($userBs->home_version,['static', 'particles', 'water', 'parallax'])){
                $rules['hero_section_image'] = [is_null($secHeading->hero_section_image) ? 'nullable' : 'nullable', new ImageMimeTypeRule()]; 
            }
            $rules['intro_img'] = [is_null($secHeading->intro_img) ? 'nullable' : 'nullable', new ImageMimeTypeRule()];
            $rules['video_section_image'] = [is_null($secHeading->video_section_image) ? 'nullable' : 'nullable', new ImageMimeTypeRule()];
            $rules['facility_section_image'] = [is_null($secHeading->facility_section_image) ? 'nullable' : 'nullable', new ImageMimeTypeRule()];
        }
        //theme version 2
        if (in_array($userBs->theme_version, [ThemeVersion::ThemeTwo->value])) {
            if (in_array($userBs->home_version, ['static', 'particles', 'water', 'parallax'])) {
                $rules['hero_section_image'] = [is_null($secHeading->hero_section_image) ? 'nullable' : 'nullable', new ImageMimeTypeRule()];
            }
            $rules['intro_img'] = [is_null($secHeading->intro_img) ? 'nullable' : 'nullable', new ImageMimeTypeRule()];
            $rules['video_section_image'] = [is_null($secHeading->video_section_image) ? 'nullable' : 'nullable', new ImageMimeTypeRule()];
            $rules['facility_section_image'] = [is_null($secHeading->facility_section_image) ? 'nullable' : 'nullable', new ImageMimeTypeRule()];
        }
        //theme version 3
        if (in_array($userBs->theme_version, [ThemeVersion::ThemeThree->value])) {
            $rules['intro_img'] = [is_null($secHeading->intro_img) ? 'nullable' : 'nullable', new ImageMimeTypeRule()];
            $rules['intro_image_two'] = [is_null($secHeading->intro_image_two) ? 'nullable' : 'nullable', new ImageMimeTypeRule()];
            $rules['intro_image_three'] = [is_null($secHeading->intro_image_three) ? 'nullable' : 'nullable', new ImageMimeTypeRule()];           
            $rules['intro_member_image'] = [is_null($secHeading->intro_member_image) ? 'nullable' : 'nullable', new ImageMimeTypeRule()];
            $rules['room_category_section_image'] = [is_null($secHeading->room_category_section_image) ? 'nullable' : 'nullable', new ImageMimeTypeRule()];
            $rules['video_section_image'] = [is_null($secHeading->video_section_image) ? 'nullable' : 'nullable', new ImageMimeTypeRule()];
            $rules['video_section_image_two'] = [is_null($secHeading->video_section_image_two) ? 'nullable' : 'nullable', new ImageMimeTypeRule()];
            $rules['statistics_section_image'] = [is_null($secHeading->statistics_section_image) ? 'nullable' : 'nullable', new ImageMimeTypeRule()];
        }
        //theme version 4
        if (in_array($userBs->theme_version, [ThemeVersion::ThemeFour->value])) {
            $rules['intro_img'] = [is_null($secHeading->intro_img) ? 'nullable' : 'nullable', new ImageMimeTypeRule()];
            $rules['intro_member_image'] = [is_null($secHeading->intro_member_image) ? 'nullable' : 'nullable', new ImageMimeTypeRule()];
            $rules['video_section_image'] = [is_null($secHeading->video_section_image) ? 'nullable' : 'nullable', new ImageMimeTypeRule()];
            $rules['facility_section_image'] = [is_null($secHeading->facility_section_image) ? 'nullable' : 'nullable', new ImageMimeTypeRule()];
        }
        
        //theme version 5
        if (in_array($userBs->theme_version, [ThemeVersion::ThemeFive->value])) {
            $rules['intro_img'] = [is_null($secHeading->intro_img) ? 'nullable' : 'nullable',new ImageMimeTypeRule()];
            $rules['intro_image_two'] = [is_null($secHeading->intro_image_two) ? 'nullable' : 'nullable',new ImageMimeTypeRule()];
            $rules['intro_member_image'] = [is_null($secHeading->intro_member_image) ? 'nullable' : 'nullable',new ImageMimeTypeRule()];     
            $rules['video_section_image'] = [is_null($secHeading->video_section_image) ? 'nullable' : 'nullable', new ImageMimeTypeRule()];
            $rules['facility_section_image'] = [is_null($secHeading->facility_section_image) ? 'nullable' : 'nullable', new ImageMimeTypeRule()];
            $rules['facility_section_image_two'] = [is_null($secHeading->facility_section_image_two) ? 'nullable' : 'nullable',new ImageMimeTypeRule()];
            $rules['facility_section_image_three'] = [is_null($secHeading->facility_section_image_three) ? 'nullable' : 'nullable',new ImageMimeTypeRule()];
            $rules['statistics_section_image'] = [is_null($secHeading->statistics_section_image) ? 'nullable' : 'nullable',new ImageMimeTypeRule()];
        }


        $messages = [];

        $validator = Validator::make($request->all(), $rules, $messages);
        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        //testimonial section image
        $testimonial_section_image = $secHeading->testimonial_section_image;
        if ($request->hasFile('testimonial_section_image')) {
            $testimonial_section_image = Uploader::update_picture(
                public_path(Constant::WEBSITE_TESTIMONIAL_SECTION_IMAGE),
                $request->file('testimonial_section_image'),
                $secHeading->testimonial_section_image
            );
        }

        //facility section image
        $facility_section_image = $secHeading->facility_section_image;
        if ($request->hasFile('facility_section_image')) {
            $facility_section_image = Uploader::update_picture(
                public_path(Constant::WEBSITE_FACILITY_SECTION_IMAGE),
                $request->file('facility_section_image'),
                $secHeading->facility_section_image
            );
        }

        $facility_section_image_two = $secHeading->facility_section_image_two;
        if ($request->hasFile('facility_section_image_two')) {
            $facility_section_image_two = Uploader::update_picture(
                public_path(Constant::WEBSITE_FACILITY_SECTION_IMAGE),
                $request->file('facility_section_image_two'),
                $secHeading->facility_section_image_two
            );
        }

        $facility_section_image_three = $secHeading->facility_section_image_three;
        if ($request->hasFile('facility_section_image_three')) {
            $facility_section_image_three = Uploader::update_picture(
                public_path(Constant::WEBSITE_FACILITY_SECTION_IMAGE),
                $request->file('facility_section_image_three'),
                $secHeading->facility_section_image_three
            );
        }


        //faq section image
        $faq_section_image = $secHeading->faq_section_image;
        if ($request->hasFile('faq_section_image')) {
            $faq_section_image = Uploader::update_picture(
                public_path(Constant::WEBSITE_FAQ_SECTION_IMAGE),
                $request->file('faq_section_image'),
                $secHeading->faq_section_image
            );
        }

        //intro image
        $intro_img = $secHeading->intro_img;
        if ($request->hasFile('intro_img')) {
            $intro_img = Uploader::update_picture(
                public_path(Constant::WEBSITE_INTRO_SECTION_IMAGE),
                $request->file('intro_img'),
                $secHeading->intro_img
            );
        }

        $intro_image_two = $secHeading->intro_image_two;
        if ($request->hasFile('intro_image_two')) {
            $intro_image_two = Uploader::update_picture(
                public_path(Constant::WEBSITE_INTRO_SECTION_IMAGE),
                $request->file('intro_image_two'),
                $secHeading->intro_image_two
            );
        }

        $intro_image_three = $secHeading->intro_image_three;
        if ($request->hasFile('intro_image_three')) {
            $intro_image_three = Uploader::update_picture(
                public_path(Constant::WEBSITE_INTRO_SECTION_IMAGE),
                $request->file('intro_image_three'),
                $secHeading->intro_image_three
            );
        }

        $intro_member_image = $secHeading->intro_member_image;
        if ($request->hasFile('intro_member_image')) {
            $intro_member_image = Uploader::update_picture(
                public_path(Constant::WEBSITE_INTRO_SECTION_IMAGE),
                $request->file('intro_member_image'),
                $secHeading->intro_member_image
            );
        }


        //hero section image
        $hero_section_image = $secHeading->hero_section_image;
        if ($request->hasFile('hero_section_image')) {
            $hero_section_image = Uploader::update_picture(
                public_path(Constant::WEBSITE_HERO_SECTION_IMAGE),
                $request->file('hero_section_image'),
                $secHeading->hero_section_image
            );
        }

        $hero_section_video_link = $request->hero_section_video_link;
        if (strpos($hero_section_video_link, '&') != 0) {
            $hero_section_video_link = substr($hero_section_video_link, 0, strpos($hero_section_video_link, '&'));
        }


        $video_section_image = $secHeading->video_section_image;
        if ($request->hasFile('video_section_image')) {
            $video_section_image = Uploader::update_picture(
                public_path(Constant::WEBSITE_VIDEO_SECTION_IMAGE),
                $request->file('video_section_image'),
                $secHeading->video_section_image
            );
        }

        $video_section_image_two = $secHeading->video_section_image_two;
        if ($request->hasFile('video_section_image_two')) {
            $video_section_image_two = Uploader::update_picture(
                public_path(Constant::WEBSITE_VIDEO_SECTION_IMAGE),
                $request->file('video_section_image_two'),
                $secHeading->video_section_image_two
            );
        }


        $room_category_section_image = $secHeading->room_category_section_image;
        if ($request->hasFile('room_category_section_image')) {
            $room_category_section_image = Uploader::update_picture(
                public_path(Constant::WEBSITE_ROOM_IMAGE),
                $request->file('room_category_section_image'),
                $secHeading->room_category_section_image
            );
        }



        $video_section_video_url = $request->video_section_video_url;
        if (strpos($video_section_video_url, '&') != 0) {
            $video_section_video_url = substr($video_section_video_url, 0, strpos($video_section_video_url, '&'));
        }

        $statistics_section_image = $secHeading->statistics_section_image;
        if ($request->hasFile('statistics_section_image')) {
            $statistics_section_image = Uploader::update_picture(public_path(Constant::WEBSITE_STATISTICS_SECTION_IMAGE),$request->file('statistics_section_image'),$secHeading->statistics_section_image);
        }

        SectionHeading::query()->updateOrInsert(
            ['user_id' => Auth::guard('web')->user()->id, 'language_id' => $languageId],
            [
                'room_section_title' =>  $request->has('room_section_title') ? $request->room_section_title : optional($secHeading)->room_section_title ,
                'room_category_section_title' =>  $request->has('room_category_section_title') ? $request->room_category_section_title : optional($secHeading)->room_category_section_title ,
                
                'room_section_subtitle' =>  $request->has('room_section_subtitle') ? $request->room_section_subtitle : optional($secHeading)->room_section_subtitle,
                'room_section_text' =>  $request->has('room_section_text') ? Purifier::clean($request->room_section_text, 'youtube') : optional($secHeading)->room_section_text,

                'service_section_title' =>  $request->has('service_section_title') ? $request->service_section_title : optional($secHeading)->service_section_title,

                'service_section_subtitle' =>  $request->has('service_section_subtitle') ? $request->service_section_subtitle : optional($secHeading)->service_section_subtitle,


                'video_section_title' =>  $request->has('video_section_title') ? $request->video_section_title : optional($secHeading)->video_section_title,

                'video_section_subtitle' =>  $request->has('video_section_subtitle') ? $request->video_section_subtitle : optional($secHeading)->video_section_subtitle,

                'video_section_button' =>  $request->has('video_section_button') ? $request->video_section_button : optional($secHeading)->video_section_button,

                'video_section_button_url' =>  $request->has('video_section_button_url') ? $request->video_section_button_url : optional($secHeading)->video_section_button_url,

                'video_section_video_url' =>  $request->has('video_section_video_url') ? $request->video_section_video_url : optional($secHeading)->video_section_video_url,

                'package_section_title' =>  $request->has('package_section_title') ? $request->package_section_title : optional($secHeading)->package_section_title,

                'package_section_subtitle' =>  $request->has('package_section_subtitle') ? $request->package_section_subtitle : optional($secHeading)->package_section_subtitle,
                
                'facility_section_title' =>  $request->has('facility_section_title') ? $request->facility_section_title : optional($secHeading)->facility_section_title,

                'facility_section_subtitle' =>  $request->has('facility_section_subtitle') ? $request->facility_section_subtitle : optional($secHeading)->facility_section_subtitle,

                'testimonial_section_title' =>  $request->has('testimonial_section_title') ? $request->testimonial_section_title : optional($secHeading)->testimonial_section_title,

                'testimonial_section_subtitle' =>  $request->has('testimonial_section_subtitle') ? $request->testimonial_section_subtitle : optional($secHeading)->testimonial_section_subtitle,

                'faq_section_title' =>  $request->has('faq_section_title') ? $request->faq_section_title : optional($secHeading)->faq_section_title,
                
                'faq_section_subtitle' =>  $request->has('faq_section_subtitle') ? $request->faq_section_subtitle : optional($secHeading)->faq_section_subtitle,
                
                'blog_section_title' =>  $request->has('blog_section_title') ? $request->blog_section_title : optional($secHeading)->blog_section_title,
                
                'blog_section_subtitle' =>  $request->has('blog_section_subtitle') ? $request->blog_section_subtitle : optional($secHeading)->blog_section_subtitle,

                'intro_primary_title' =>  $request->has('intro_primary_title') ? $request->intro_primary_title : optional($secHeading)->intro_primary_title,
                
                'intro_secondary_title' =>  $request->has('intro_secondary_title') ? $request->intro_secondary_title : optional($secHeading)->intro_secondary_title,
                
                'intro_text' =>  $request->has('intro_text') ? Purifier::clean($request->intro_text, 'youtube') : optional($secHeading)->intro_text,
                'intro_button_url' =>  $request->has('intro_button_url') ? $request->intro_button_url : optional($secHeading)->intro_button_url,
                'intro_button_text' =>  $request->has('intro_button_text') ? $request->intro_button_text : optional($secHeading)->intro_button_text,
                
                'hero_section_title' =>  $request->has('hero_section_title') ? $request->hero_section_title : optional($secHeading)->hero_section_title,
                
                'hero_section_subtitle' =>  $request->has('hero_section_subtitle') ? $request->hero_section_subtitle : optional($secHeading)->hero_section_subtitle,
                'hero_section_button_name' =>  $request->has('hero_section_button_name') ? $request->hero_section_button_name : optional($secHeading)->hero_section_button_name,

                'hero_section_button_url' =>  $request->has('hero_section_button_url') ? $request->hero_section_button_url : optional($secHeading)->hero_section_button_url,


                'hero_section_video_link' =>  $request->has('hero_section_video_link') ? $request->hero_section_video_link : optional($secHeading)->hero_section_video_link,

                'latest_room_section_title' =>  $request->has('latest_room_section_title') ? $request->latest_room_section_title : optional($secHeading)->latest_room_section_title,

                'latest_package_section_title' =>  $request->has('latest_package_section_title') ? $request->latest_package_section_title : optional($secHeading)->latest_package_section_title,

                'intro_member_image_show' =>  $request->has('intro_member_image_show') ? $request->intro_member_image_show : optional($secHeading)->intro_member_image_show,

                'package_category_section_title' =>  $request->has('package_category_section_title') ? $request->package_category_section_title : optional($secHeading)->package_category_section_title,


                'hero_section_image' => $hero_section_image,
                'faq_section_image' => $faq_section_image,
                'facility_section_image' => $facility_section_image,
                'testimonial_section_image' => $testimonial_section_image,
                'intro_img' => $intro_img,
                'video_section_image' => $video_section_image,
                'video_section_image_two' => $video_section_image_two,
                'statistics_section_image' => $statistics_section_image,

                'intro_image_two' => $intro_image_two,
                'intro_image_three' => $intro_image_three,
                'intro_member_image' => $intro_member_image,
                'facility_section_image_two' => $facility_section_image_two,
                'facility_section_image_three' => $facility_section_image_three,
                'room_category_section_image' => $room_category_section_image,
            ]
        );
        Session::flash('success', __('Updated Successfully'));
        return "success";
    }

    public function error_404(Request $request)
    {
        $language = Language::where('code', $request->language)
            ->where('user_id', Auth::guard('web')->user()->id)
            ->firstOrFail();

        $data['data'] = BasicExtended::query()
            ->where('user_id', Auth::guard('web')->user()->id)
            ->where('language_id', $language->id)
            ->first();
        return view('user.settings.error_404', $data);
    }
    public function updateError_404(Request $request)
    {
        $language = Language::where('code', $request->language)->where('user_id', Auth::guard('web')->user()->id)->firstOrFail();
        $userBe = BasicExtended::query()
            ->where('user_id', Auth::guard('web')->user()->id)
            ->where('language_id', $language->id)
            ->first();

        $rules = [
            'page_not_found_title' => 'required | max:255',
            'page_not_found_subtitle' => 'required | max:255',
            'page_not_found_image_one' => [is_null($userBe->page_not_found_image_one) ? 'required' : 'nullable', new ImageMimeTypeRule()],
            'page_not_found_image_two' => [is_null($userBe->page_not_found_image_two) ? 'required' : 'nullable', new ImageMimeTypeRule()],
        ];
        $validator = Validator::make($request->all(), $rules);
        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        $page_not_found_image_one = $userBe->page_not_found_image_one;
        if ($request->hasFile('page_not_found_image_one')) {
            $page_not_found_image_one = Uploader::update_picture(
                public_path(Constant::WEBSITE_ERROR_IMAGE),
                $request->file('page_not_found_image_one'),
                $userBe->page_not_found_image_one
            );
        }

        $page_not_found_image_two = $userBe->page_not_found_image_two;
        if ($request->hasFile('page_not_found_image_two')) {
            $page_not_found_image_two = Uploader::update_picture(
                public_path(Constant::WEBSITE_ERROR_IMAGE),
                $request->file('page_not_found_image_two'),
                $userBe->page_not_found_image_two
            );
        }


        BasicExtended::query()->updateOrInsert(
            ['user_id' => Auth::guard('web')->user()->id, 'language_id' => $language->id],
            [
                'page_not_found_title' => $request->page_not_found_title,
                'page_not_found_subtitle' => $request->page_not_found_subtitle,
                'page_not_found_image_one' => $page_not_found_image_one,
                'page_not_found_image_two' => $page_not_found_image_two,
            ]
        );

        Session::flash('success', __('Updated Successfully'));
        return "success";
    }

    public function loginPage(Request $request)
    {
        $language = Language::where('code', $request->language)
            ->where('user_id', Auth::guard('web')->user()->id)
            ->firstOrFail();

        $data['data'] = BasicExtended::query()
            ->where('user_id', Auth::guard('web')->user()->id)
            ->where('language_id', $language->id)
            ->first();
        return view('user.settings.login_page', $data);
    }

    public function updateloginPage(Request $request)
    {
        $language = Language::where('code', $request->language)->where('user_id', Auth::guard('web')->user()->id)->firstOrFail();
        $userBe = BasicExtended::query()
            ->where('user_id', Auth::guard('web')->user()->id)
            ->where('language_id', $language->id)
            ->first();

        $rules = [
            'login_page_title' => 'required | max:255',
            'login_page_subtitle' => 'nullable | max:255',
            'login_page_image' => [is_null($userBe->login_page_image) ? 'required' : 'nullable', new ImageMimeTypeRule()],
        ];
        $validator = Validator::make($request->all(), $rules);
        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }
        $login_page_image = $userBe->login_page_image;
        
        if ($request->hasFile('login_page_image')) {
            $login_page_image = Uploader::update_picture(
                public_path(Constant::WEBSITE_IMAGE),
                $request->file('login_page_image'),
                $userBe->login_page_image
            );
        }

        BasicExtended::query()->updateOrInsert(
            ['user_id' => Auth::guard('web')->user()->id, 'language_id' => $language->id],
            [
                'login_page_title' => $request->login_page_title,
                'login_page_subtitle' => $request->login_page_subtitle,
                'login_page_image' => $login_page_image,
            ]
        );

        Session::flash('success', __('Updated Successfully'));
        return "success";
    }

    public function signupPage(Request $request)
    {
        $language = Language::where('code', $request->language)
            ->where('user_id', Auth::guard('web')->user()->id)
            ->firstOrFail();

        $data['data'] = BasicExtended::query()
            ->where('user_id', Auth::guard('web')->user()->id)
            ->where('language_id', $language->id)
            ->first();
        return view('user.settings.signup_page', $data);
    }

    public function updatesignupPage(Request $request)
    {
        $language = Language::where('code', $request->language)->where('user_id', Auth::guard('web')->user()->id)->firstOrFail();
        $userBe = BasicExtended::query()
            ->where('user_id', Auth::guard('web')->user()->id)
            ->where('language_id', $language->id)
            ->first();

        $rules = [
            'signup_page_title' => 'required | max:255',
            'signup_page_subtitle' => 'nullable | max:255',
            'signup_page_image' => [is_null($userBe->signup_page_image) ? 'required' : 'nullable', new ImageMimeTypeRule()],
        ];
        $validator = Validator::make($request->all(), $rules);
        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }
        $signup_page_image = $userBe->signup_page_image;

        if ($request->hasFile('signup_page_image')) {
            $signup_page_image = Uploader::update_picture(
                public_path(Constant::WEBSITE_IMAGE),
                $request->file('signup_page_image'),
                $userBe->signup_page_image
            );
        }

        BasicExtended::query()->updateOrInsert(
            ['user_id' => Auth::guard('web')->user()->id, 'language_id' => $language->id],
            [
                'signup_page_title' => $request->signup_page_title,
                'signup_page_subtitle' => $request->signup_page_subtitle,
                'signup_page_image' => $signup_page_image,
            ]
        );

        Session::flash('success', __('Updated Successfully'));
        return "success";
    }
}
