<?php

namespace App\Http\Controllers\User;

use App\Constants\Constant;
use App\Http\Controllers\Controller;
use App\Http\Helpers\Uploader;
use App\Http\Requests\Advertisement\StoreRequest;
use App\Http\Requests\Advertisement\UpdateRequest;
use App\Models\User\Advertisement;
use App\Models\User\BasicSetting;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Validator;

class AdvertisementController extends Controller
{

    public function advertiseSettings()
    {
        $data = BasicSetting::query()
            ->where('user_id', Auth::guard('web')->user()->id)
            ->select('google_adsense_publisher_id')
            ->first();
        return view('user.advertisement.settings', ['data' => $data]);
    }

    public function updateAdvertiseSettings(Request $request)
    {
        $rule = [
            'google_adsense_publisher_id' => 'nullable'
        ];

        $validator = Validator::make($request->all(), $rule);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator->errors());
        }

        BasicSetting::query()->updateOrInsert(
            ['user_id' => Auth::guard('web')->user()->id],
            ['google_adsense_publisher_id' => $request->google_adsense_publisher_id]
        );
        $request->session()->flash('success', __('Updated Successfully'));
        return redirect()->back();
    }
    public function index()
    {
        $ads = Advertisement::query()->where('user_id', Auth::guard('web')->user()->id)->orderByDesc('id')->get();
        return view('user.advertisement.index', compact('ads'));
    }

    public function store(StoreRequest $request): string
    {
        $imageName = null;
        if ($request->hasFile('image')) {
            $imageName = Uploader::upload_picture(
                public_path(Constant::WEBSITE_ADVERTISEMENT_IMAGE),
                $request->file('image')
            );
        }
        Advertisement::create($request->except('image') + [
            'user_id' => Auth::guard('web')->user()->id,
            'image' => $request->hasFile('image') ? $imageName : null
        ]);

        $request->session()->flash('success', __('Added Successfully'));
        return "success";
    }

    public function update(UpdateRequest $request)
    {
        $ad = Advertisement::query()->where('user_id', Auth::guard('web')->user()->id)->find($request->id);
        $imageName = null;
        if ($request->hasFile('image')) {
            $imageName = Uploader::update_picture(
                public_path(Constant::WEBSITE_ADVERTISEMENT_IMAGE),
                $request->file('image'),
                $ad->image
            );
        }
        if ($request->ad_type == 'adsense') {
            // if ad type change to google adsense then delete the image from local storage.
            Uploader::remove(
                public_path(Constant::WEBSITE_ADVERTISEMENT_IMAGE),
                $ad->image
            );
        }
        $ad->update($request->except('image') + [
            'image' => $request->hasFile('image') ? $imageName : $ad->image
        ]);
        $request->session()->flash('success', __('Updated Successfully'));
        return "success";
    }

    public function destroy($id): \Illuminate\Http\RedirectResponse
    {
        $ad = Advertisement::query()->where('user_id', Auth::guard('web')->user()->id)->find($id);
        if ($ad->ad_type == 'banner') Uploader::remove(public_path(Constant::WEBSITE_ADVERTISEMENT_IMAGE), $ad->image);
        $ad->delete();
        return redirect()->back()->with('success', __('Deleted Successfully'));
    }

    public function bulkDestroy(Request $request): string
    {
        $ids = $request->ids;
        foreach ($ids as $id) {
            $ad = Advertisement::query()->where('user_id', Auth::guard('web')->user()->id)->find($id);
            if ($ad->ad_type == 'banner') Uploader::remove(public_path(Constant::WEBSITE_ADVERTISEMENT_IMAGE), $ad->image);
            $ad->delete();
        }
       Session::flash('success', __('Deleted Successfully'));
        return "success";
    }
}
