<?php

namespace App\Http\Controllers\FrontEnd\Room;

use App\Http\Controllers\Controller;
use App\Models\User\Language;
use App\Models\User\BasicSetting;
use App\Models\User\Coupon;
use App\Models\User\OfflineGateway;
use App\Models\User\PaymentGateway;
use App\Models\User\Room;
use App\Models\User\RoomAmenity;
use App\Models\User\RoomBooking;
use App\Models\User\RoomCategory;
use App\Models\User\RoomContent;
use App\Models\User\RoomReview;
use App\Traits\MiscellaneousTrait;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Validator;

class RoomController extends Controller
{
  use MiscellaneousTrait;

  public function rooms(Request $request)
  {
    $user = getUser();
    $queryResult['breadcrumbInfo'] = MiscellaneousTrait::getBreadcrumb();
    $language = MiscellaneousTrait::getLanguage();
    $queryResult['pageHeading'] = $this->getUserPageHeading($language, $user->id);

    $num_of_bed = $num_of_bath = $num_of_guests = $min_rent = $max_rent = null;

    $roomIds = [];
    $dates = [];

 
    
    if ($request->filled('dates')) {
      $dateArray = explode(' ', $request->dates);
      $date1 = $dateArray[0];
      $date2 = $dateArray[2];
      $dates = $this->displayDates($date1, $date2);
      $rooms = Room::where('user_id', getUser()->id)->get();
      foreach ($rooms as $key => $room) {
        foreach ($dates as $key => $date) {
          $cDate = Carbon::parse($date);
          $count = RoomBooking::whereDate('arrival_date', '<=', $cDate)->whereDate('departure_date', '>=', $cDate)->where('room_id', $room->id)->count();

          if ($count >= $room->quantity) {
            if (!in_array($room->id, $roomIds)) {
              $roomIds[] = $room->id;
            }
          }
        }
      }
    }

    if ($request->filled('beds')) {
      $num_of_bed = $request->beds;
    }
    if ($request->filled('baths')) {
      $num_of_bath = $request->baths;
    }
    if ($request->filled('guests')) {
      $num_of_guests = $request->guests;
    }
    if ($request->filled('rents')) {
      $bs = BasicSetting::select('base_currency_symbol')->where('user_id', $user->id)->first();
      $rents = str_replace($bs->base_currency_symbol, ' ', $request->rents);
      $rentArray = explode(' ', $rents);
      $min_rent = $rentArray[1];
      $max_rent = $rentArray[4];
    }

    $category = $request->category;
    $sortBy = $request->sort_by;
    $ammenities = $request->ammenities;
    $ratings = $request->ratings ?? null;
    $locations = $request->locations;

    $queryResult['categories'] = RoomCategory::where('language_id', $language->id)
      ->where('status', 1)
      ->where('user_id', $user->id)
      ->orderBy('serial_number', 'asc')
      ->get();
      
    $queryResult['roomInfos'] = DB::table('user_rooms')
      ->join('user_room_contents', 'user_rooms.id', '=', 'user_room_contents.room_id')
      ->join('user_room_categories', 'user_rooms.category_index', '=', 'user_room_categories.indx')
      ->where('user_room_categories.language_id', '=', $language->id)
      ->where('user_rooms.status', '=', 1)
      ->where('user_rooms.user_id', '=', $user->id)
      ->where('user_room_contents.language_id', '=', $language->id)
      ->when($category, function ($query) use ($category, $language, $user) {
        $cat = RoomCategory::where('language_id', $language->id)
          ->where('user_id', $user->id)
          ->where('slug', $category)
          ->first();
        if (!is_null($cat)) {
          return $query->where('user_rooms.category_index', $cat->indx);
        }
      })->when($num_of_guests, function ($query, $num_of_guests) {
        if ($num_of_guests == 10) {
          return $query->where('max_guests', $num_of_guests);
        } else {
          return $query->where('max_guests', '>=', $num_of_guests);
        }
      })->when($num_of_bed, function ($query, $num_of_bed) {
        return $query->where('bed', $num_of_bed);
      })->when($ratings, function ($query, $ratings) {
        return $query->where('avg_rating', '>=', $ratings);
      })->when($locations, function ($query, $locations) {
        return $query->where('address', 'like', '%' . $locations . '%');
      })->when($num_of_bath, function ($query, $num_of_bath) {
        return $query->where('bath', $num_of_bath);
      })->when(($min_rent && $max_rent), function ($query) use ($min_rent, $max_rent) {
        return $query->where('rent', '>=', $min_rent)->where('rent', '<=', $max_rent);
      })->when($ammenities, function ($query, $ammenities) {
        return $query->where(function ($query) use ($ammenities) {
          foreach ($ammenities as $key => $amm) {
            if ($key == 0) {
              $query->where('user_rooms.amenities_index', 'LIKE',  "%" . '"' . $amm . '"' . "%");
            } else {
              $query->orWhere('user_rooms.amenities_index', 'LIKE', "%" . '"' . $amm . '"' . "%");
            }
          }
        });
      })->when($sortBy, function ($query, $sortBy) {
        if ($sortBy == 'asc') {
          return $query->orderBy('user_rooms.id', 'ASC');
        } elseif ($sortBy == 'desc') {
          return $query->orderBy('user_rooms.id', 'DESC');
        } elseif ($sortBy == 'price-desc') {
          return $query->orderBy('rent', 'DESC');
        } elseif ($sortBy == 'price-asc') {
          return $query->orderBy('rent', 'ASC');
        }
      }, function ($query) {
        return $query->orderBy('user_rooms.id', 'DESC');
      })
      ->whereNotIn('user_rooms.id', $roomIds)
      ->select('user_room_contents.*', 'user_rooms.*', 'user_room_categories.indx', 'user_room_categories.name as category_name', 'user_room_categories.slug as category_slug')
      ->paginate(6);


    $queryResult['currencyInfo'] = MiscellaneousTrait::getCurrencyInfo();

    $queryResult['numOfBed'] = Room::where('user_id', $user->id)->where('status', 1)->max('bed');

    $queryResult['numOfBath'] = Room::where('user_id', $user->id)->where('status', 1)->max('bath');

    $maxPrice = Room::where('user_id', $user->id)->where('status', 1)->max('rent');
    $minPrice = Room::where('user_id', $user->id)->where('status', 1)->min('rent');
    $maxGuests = Room::where('user_id', $user->id)->where('status', 1)->max('max_guests');

    $queryResult['maxPrice'] = $maxPrice;
    $queryResult['minPrice'] = $minPrice;
    $queryResult['maxGuests'] = $maxGuests;

    if ($request->filled('rents')) {
      $queryResult['maxRent'] = $max_rent;
      $queryResult['minRent'] = $min_rent;
    } else {
      $queryResult['maxRent'] = $maxPrice;
      $queryResult['minRent'] = $minPrice;
    }

    $queryResult['amenities'] = RoomAmenity::where('language_id', $language->id)->where('user_id', $user->id)->where('status',1)->orderBy('serial_number','asc')->get();
    if ($request->ajax() && $request->sidebar_search == 1) {
      return view('user-front.common.room.grid_view', $queryResult)->render();
    }
    $queryResult['seo'] = $this->getUserSeoData($language, $user->id);
    return view('user-front.common.room.rooms', $queryResult);
  }

  public function displayDates($date1, $date2, $format = 'Y-m-d')
  {
    $dates = array();
    $current = strtotime($date1);
    $date2 = strtotime($date2);
    $stepVal = '+1 day';

    while ($current <= $date2) {
      $dates[] = date($format, $current);
      $current = strtotime($stepVal, $current);
    }

    return $dates;
  }

  public function roomDetails($domain, $slug)
  {
    $user = getuser();
    $queryResult['breadcrumbInfo'] = MiscellaneousTrait::getBreadcrumb();
    $language = MiscellaneousTrait::getLanguage();
    $details = RoomContent::join('user_rooms', 'user_rooms.id', 'user_room_contents.room_id')
      ->where('language_id', $language->id)
      ->where('user_rooms.user_id', '=', $user->id)
      ->where('user_rooms.status', '=', 1)
      ->where('slug', $slug)
      ->firstorFail();
      
    $id = $details->room_id;
    $queryResult['details'] = $details;
    $amms = [];
    if (!empty($details->amenities_index) && $details->amenitiesamenities_index != '[]') {
      $ammIds = json_decode($details->amenities_index, true);
      
      $ammenities = RoomAmenity::whereIn('indx', $ammIds)->where('user_id',$user->id)->where('language_id',$language->id)->where('status',1)->orderBy('serial_number', 'ASC')->get();
      foreach ($ammenities as $key => $ammenity) {
        $amms[] = $ammenity->name;
      }
    }
  
    $queryResult['amms'] = $amms;
    $queryResult['reviews'] = RoomReview::where('room_id', $id)
    ->where('user_id',$user->id)
    ->orderBy('id', 'DESC')->get();
    $queryResult['currencyInfo'] = MiscellaneousTrait::getCurrencyInfo();
    $bookings = RoomBooking::where('room_id', $id)
     ->where('user_id',$user->id)
      ->select('id', 'arrival_date', 'departure_date')
      ->where('payment_status', 1)
      ->with('roomBookedByCustomer')
      ->get();

    $qty = Room::findOrFail($id)->quantity;
    $bookedDates = [];
    foreach ($bookings as $key => $booking) {
      // get all dates of a booking date range
      $dates = [];
      $dates = $this->displayDates($booking->arrival_date, $booking->departure_date);

      // loop through the dates
      foreach ($dates as $key => $date) {
        $count = 1;
        foreach ($bookings as $key => $cbooking) {
          if ($cbooking->id != $booking->id) {
            $start = Carbon::parse($cbooking->arrival_date);
            $departure = Carbon::parse($cbooking->departure_date);
            $cDate = Carbon::parse($date);

            // check if the date is present in other booking's date ranges
            if ($cDate->gte($start) && $cDate->lte($departure)) {
              $count++;
            }
          }
        }
        // number of booking of a date is equal to room quantity, then mark the date as booked
        if ($count >= $qty && !in_array($date, $bookedDates)) {
          $bookedDates[] = $date;
        }
      }
    }

    $queryResult['bookingDates'] = $bookedDates;
    $queryResult['onlineGateways'] = PaymentGateway::where('status', 1)->where('user_id', getUser()->id)->get();
    $queryResult['offlineGateways'] = OfflineGateway::where('user_id', $user->id)->orderBy('serial_number', 'asc')->get()->map(function ($gateway) {
      return [
        'id' => $gateway->id,
        'name' => $gateway->name,
        'short_description' => $gateway->short_description,
        'instructions' => replaceBaseUrl($gateway->instructions, 'summernote'),
        'is_receipt' => $gateway->is_receipt,
        'serial_number' => $gateway->serial_number
      ];
    });

    $queryResult['latestRooms'] = RoomContent::where('language_id', $language->id)
      ->whereHas('room', function ($query) {
        $query->status();
      })
      ->with('room')
      ->where('room_id', '<>', $id)
      ->where('user_id', $user->id)
      ->orderBy('room_id', 'desc')
      ->limit(3)
      ->get();

    if (Auth::guard('customer')->check()) {
      $queryResult['customer'] = RoomBooking::where('user_id',$user->id)->where('customer_id', Auth::guard('customer')->user()->id)->with('roomBookedByCustomer')->first();
    }

    $queryResult['avgRating'] = RoomReview::where('room_id', $id)->where('user_id', $user->id)->avg('rating');

    $stripe = PaymentGateway::query()
      ->where('keyword', 'stripe')
      ->where('user_id', $user->id)
      ->first();

    if (!is_null($stripe->information)) {
      $stripe = json_decode($stripe->information, true);
      $queryResult['stripe_key'] = $stripe['key'];
    }

    $queryResult['anet'] = PaymentGateway::query()
      ->where('keyword', 'authorize.net')
      ->where('user_id', getUser()->id)
      ->first();

    $queryResult['seo'] = $this->getUserSeoData($language, $user->id);

    return view('user-front.common.room.room_details', $queryResult);
  }

  public function applyCoupon(Request $request)
  {
    $keywords = get_keywords(getUser()->id);
    try {
      $coupon = Coupon::where('code', $request->coupon)->firstOrFail();

      $startDate = Carbon::parse($coupon->start_date);
      $endDate = Carbon::parse($coupon->end_date);
      $todayDate = Carbon::now();

      // check coupon is valid or not
      if ($todayDate->between($startDate, $endDate) == false) {
        return response()->json(['error' => $keywords['Sorry, coupon has been expired!'] ?? __('Sorry, coupon has been expired!')]);
      }

      // check coupon is valid or not for this room
      $roomId = $request->roomId;
      $roomIds = empty($coupon->rooms) ? '' : json_decode($coupon->rooms);

      if (!empty($roomIds) && !in_array($roomId, $roomIds)) {
        return response()->json(['error' => $keywords['You can not apply this coupon for this room!'] ?? __('You can not apply this coupon for this room!')]);
      }

      Session::put('couponCode', $request->coupon);

      $initTotalRent = str_replace(',', '', $request->initTotal);

      if ($initTotalRent == '0.00') {
        return response()->json(['error' => $keywords['First, fillup the booking dates.'] ?? __('First, fillup the booking dates.')]);
      } else {
        if ($coupon->type == 'fixed') {
          $total = floatval($initTotalRent) - floatval($coupon->value);

          return response()->json([
            'success' => $keywords['Coupon applied successfully.'] ?? __('Coupon applied successfully.'),
            'discount' => $coupon->value,
            'total' => $total,
          ]);
        } else {
          $initTotalRent = floatval($initTotalRent);
          $couponVal = floatval($coupon->value);

          $discount = $initTotalRent * ($couponVal / 100);
          $total = $initTotalRent - $discount;

          return response()->json([
            'success' => $keywords['Coupon applied successfully.'] ?? __('Coupon applied successfully.'),
            'discount' => $discount,
            'total' => $total
          ]);
        }
      }
    } catch (ModelNotFoundException $e) {
      return response()->json(['error' => $keywords['Coupon is not valid!'] ?? __('Coupon is not valid!')]);
    }
  }

  public function storeReview(Request $request, $domain, $id)
  {
    $keywords = get_keywords(getUser()->id);
    $booking = RoomBooking::where('user_id', Auth::guard('customer')->user()->user_id)->where('room_id', $id)->where('payment_status', 1)->count();

    if ($booking == 0) {
      Session::flash('error', $keywords['You had not booked this room yet.'] ?? __("You had not booked this room yet."));

      return back();
    }

    $rules = ['rating' => 'required|numeric'];

    $message = [
      'rating.required' => $keywords['The star rating field is required.'] ?? __('The star rating field is required.')
    ];

    $validator = Validator::make($request->all(), $rules, $message);

    if ($validator->fails()) {
      return redirect()->back()->withErrors($validator)->withInput();
    }

    $customer = Auth::guard('customer')->user();

    $review = RoomReview::where('user_id', $customer->user_id)->where('room_id', $id)->first();

    /**
     * if, room review of auth user does not exist then create a new one.
     * otherwise, update the existing review of that auth user.
     */
    if ($review == null) {
      RoomReview::create($request->except('user_id', 'room_id') + [
        'user_id' => $customer->user_id,
        'room_id' => $id,
        'customer_id' => Auth::guard('customer')->user()->id
      ]);

      // now, store the average rating of this room
      $room = Room::where('user_id', getUser()->id)->findOrFail($id);

      $room->update(['avg_rating' => $request->rating]);
    } else {
      $review->update($request->all());

      // now, get the average rating of this room
      $roomReviews = RoomReview::where('room_id', $id)->get();

      $totalRating = 0;

      foreach ($roomReviews as $roomReview) {
        $totalRating += $roomReview->rating;
      }

      $avgRating = $totalRating / $roomReviews->count();

      // finally, store the average rating of this room
      $room = Room::findOrFail($id);

      $room->update(['avg_rating' => $avgRating]);
    }

    Session::flash('success', $keywords['Review saved successfully!'] ?? __('Review saved successfully!'));

    return redirect()->back();
  }
}
