<?php

namespace App\Http\Controllers\FrontEnd\Room;

use App\Http\Controllers\Controller;
use App\Http\Controllers\FrontEnd\Room\RoomBookingController;
use App\Http\Helpers\LimitCheckerHelper;
use App\Models\User\PaymentGateway;
use App\Models\User\RoomBooking;
use App\Traits\MiscellaneousTrait;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Session;
use Razorpay\Api\Api;
use Razorpay\Api\Errors\SignatureVerificationError;

class RazorpayController extends Controller
{
  use MiscellaneousTrait;

  private $key, $secret, $api;
  private $paypalData;

  public function __construct()
  {
      $data = PaymentGateway::whereKeyword('razorpay')->where('user_id', getUser()->id)->first();
      $razorpayData = $this->paypalData = json_decode($data->information, true);

      $this->key = $razorpayData['key'];
      $this->secret = $razorpayData['secret'];
      $this->api = new Api($this->key, $this->secret);

  }

  public function bookingProcess(Request $request)
  {
    if (empty($this->paypalData)) {
      return back()->with('error', __('Payment credentails not set yet'));
    }

    $countRoomBooking = LimitCheckerHelper::roomBookingCountUser(getUser()->id);
    $roomBookingLimit = LimitCheckerHelper::roomBookingsLimit(getUser()->id);

    if ($roomBookingLimit > $countRoomBooking) {
      $roomBooking = new RoomBookingController();

      // do calculation
      $calculatedData = $roomBooking->calculation($request);

      $currencyInfo = MiscellaneousTrait::getCurrencyInfo();
        // checking whether the currency is set to 'INR' or not
        if ($currencyInfo->base_currency_text !== 'INR') {
          return redirect()->back()->with('error', __('Invalid currency for razorpay payment.'));
        }

        $information['subtotal'] = $calculatedData['subtotal'];
        $information['discount'] = $calculatedData['discount'];
        $information['total'] = $calculatedData['total'];
        $information['userTotal'] = $calculatedData['userTotal'];
        $information['tax']       = $calculatedData['tax'];
        $information['fee']       = $calculatedData['fee'];
        $information['currency_symbol'] = $currencyInfo->base_currency_symbol;
        $information['currency_symbol_position'] = $currencyInfo->base_currency_symbol_position;
        $information['currency_text'] = $currencyInfo->base_currency_text;
        $information['currency_text_position'] = $currencyInfo->base_currency_text_position;
        $information['method'] = 'Razorpay';
        $information['type'] = 'online';

        // store the room booking information in database
        $booking_details = $roomBooking->storeData($request, $information);

        $notify_url = route('room_booking.razorpay.notify', getParam());

        $orderData = [
          'receipt'         => 'Room Booking',
          'amount'          => intval($calculatedData['total']),
          'currency'        => 'INR',
          'payment_capture' => 1 // auto capture
        ];

        $razorpayOrder = $this->api->order->create($orderData);

        $data = [
          'key'               => $this->key,
          'amount'            => $orderData['amount'],
          'name'              => $orderData['receipt'],
          'description'       => 'Booking Room via Razorpay Gateway',
          'prefill'           => [
            'name'              => $booking_details->customer_name,
            'email'             => $booking_details->customer_email,
            'contact'           => $booking_details->customer_phone
          ],
          'notes'             => [
            'merchant_order_id' => $booking_details->order_number
          ],
          'order_id'          => $razorpayOrder['id']
        ];

        $jsonData = json_encode($data);

        // put some data in session before redirect to razorpay url
        Session::put('bookingId', $booking_details->id);   // db row number
        Session::put('razorpayOrderId', $razorpayOrder['id']);

        return view('user-front.common.partials.razorpay', compact('jsonData', 'notify_url'));

    } else {
      return redirect()->back()->with('error', __('Please Contact Support'));
    }
  }

  public function notify(Request $request)
  {
    // get the information from session
    $bookingId = Session::get('bookingId');
    $razorpayOrderId = Session::get('razorpayOrderId');

    // get the information from the url, which has send by razorpay through post request
    $urlInfo = $request->all();

    // let, assume that the transaction was successfull
    $success = true;

    // Either razorpay_order_id or razorpay_subscription_id must be present
    // the keys of $attributes array must be follow razorpay convention
    try {
      $attributes = [
        'razorpay_order_id' => $razorpayOrderId,
        'razorpay_payment_id' => $urlInfo['razorpayPaymentId'],
        'razorpay_signature' => $urlInfo['razorpaySignature']
      ];

      $this->api->utility->verifyPaymentSignature($attributes);
    } catch (SignatureVerificationError $e) {
      $success = false;
    }

    if ($success === true) {
      // update the payment status for room booking in database
      $bookingInfo = RoomBooking::findOrFail($bookingId);
      $bookingInfo->update(['payment_status' => 1]);
      $roomBooking = new RoomBookingController();
      // generate an invoice in pdf format
      $invoice = $roomBooking->generateInvoice($bookingInfo);
      // update the invoice field information in database
      $bookingInfo->update(['invoice' => $invoice]);
      // send a mail to the customer with an invoice
      $roomBooking->sendMail($bookingInfo);

      // remove all session data
      Session::forget('bookingId');
      Session::forget('razorpayOrderId');
      return redirect()->route('room_booking.complete', getParam());
    } else {
      return redirect()->route('room_booking.cancel', getParam());
    }
  }
}
