<?php

namespace App\Http\Controllers\FrontEnd\Room;

use App\Http\Controllers\Controller;
use App\Http\Controllers\FrontEnd\Room\RoomBookingController;
use App\Http\Helpers\LimitCheckerHelper;
use App\Models\User\Language;
use App\Models\User\PaymentGateway;
use App\Models\User\RoomBooking;
use App\Traits\MiscellaneousTrait;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Session;

class PaystackController extends Controller
{
  use MiscellaneousTrait;

  private $api_key;
  private $adminBe;
  private $adminBs;
  private $paypalData;

  public function __construct()
  {

    $currentLang = session()->has('user_front_lang') ?
      (Language::where('code', session()->get('user_front_lang'))->first())
      : (Language::where('is_default', 1)->where('user_id', getUser()->id)->first());

    $bs = $currentLang->basic_setting;
    $be = $currentLang->basic_extended;
    $this->adminBe = $be;
    $this->adminBs = $bs;

    $data = PaymentGateway::whereKeyword('paystack')->where('user_id', getUser()->id)->first();
    $paystackData = $this->paypalData = json_decode($data->information, true);
    $this->api_key = $paystackData['key'];
  }

  public function bookingProcess(Request $request)
  {
    if (empty($this->paypalData)) {
      return back()->with('error', __('Payment credentails not set yet'));
    }

    $countRoomBooking = LimitCheckerHelper::roomBookingCountUser(getUser()->id);;
    $roomBookingLimit = LimitCheckerHelper::roomBookingsLimit(getUser()->id);

    if ($roomBookingLimit > $countRoomBooking) {
      $roomBooking = new RoomBookingController();
      $calculatedData = $roomBooking->calculation($request);
      // convert the total rent into smallest unit and convert it to an integer value
      $totalRent = intval($calculatedData['total'] * 100);


      $currencyInfo = MiscellaneousTrait::getCurrencyInfo();

      // checking whether the currency is set to 'NGN' or not
      if ($currencyInfo->base_currency_text !== 'NGN') {
        return redirect()->back()->with('error', __('Invalid currency for paystack payment.'));
      }

      $information['subtotal'] = $calculatedData['subtotal'];
      $information['discount'] = $calculatedData['discount'];
      $information['total'] = $calculatedData['total'];
      $information['userTotal'] = $calculatedData['userTotal'];
      $information['tax']       = $calculatedData['tax'];
      $information['fee']       = $calculatedData['fee'];
      $information['currency_symbol'] = $currencyInfo->base_currency_symbol;
      $information['currency_symbol_position'] = $currencyInfo->base_currency_symbol_position;
      $information['currency_text'] = $currencyInfo->base_currency_text;
      $information['currency_text_position'] = $currencyInfo->base_currency_text_position;
      $information['method'] = 'Paystack';
      $information['type'] = 'online';

      // store the room booking information in database
      $booking_details = $roomBooking->storeData($request, $information);

      $notify_url = route('room_booking.paystack.notify', getParam());

      $curl = curl_init();

      curl_setopt_array($curl, array(
        CURLOPT_URL            => 'https://api.paystack.co/transaction/initialize',
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_CUSTOMREQUEST  => 'POST',
        CURLOPT_POSTFIELDS     => json_encode([
          'amount'       => $totalRent,
          'email'        => $booking_details->customer_email,
          'callback_url' => $notify_url
        ]),
        CURLOPT_HTTPHEADER     => [
          'authorization: Bearer ' . $this->api_key,
          'content-type: application/json',
          'cache-control: no-cache'
        ]
      ));

      $response = curl_exec($curl);
      curl_close($curl);
      $transaction = json_decode($response, true);

      // put some data in session before redirect to paystack url
      Session::put('bookingId', $booking_details->id);   // db row number
      if ($transaction['status'] == true) {
        return redirect($transaction['data']['authorization_url']);
      } else {
        return redirect()->back()->with('error', __('Sorry, transaction failed!'));
      }
    } else {
      return redirect()->back()->with('error', __('Please Contact Support'));
    }
  }

  public function notify(Request $request)
  {
    // get the information from session
    $bookingId = Session::get('bookingId');

    $urlInfo = $request->all();

    if ($urlInfo['trxref'] === $urlInfo['reference']) {
      // update the payment status for room booking in database
      $bookingInfo = RoomBooking::findOrFail($bookingId);

      $bookingInfo->update(['payment_status' => 1]);

      $roomBooking = new RoomBookingController();

      // generate an invoice in pdf format
      $invoice = $roomBooking->generateInvoice($bookingInfo);

      // update the invoice field information in database
      $bookingInfo->update(['invoice' => $invoice]);

      // send a mail to the customer with an invoice
      $roomBooking->sendMail($bookingInfo);

      // remove all session data
      Session::forget('bookingId');
      return redirect()->route('room_booking.complete', getParam());
    } else {
      return redirect()->route('room_booking.cancel', getParam());
    }
  }
}
