<?php
namespace App\Http\Controllers\FrontEnd\Room;

use App\Http\Controllers\Controller;
use App\Http\Controllers\FrontEnd\Room\RoomBookingController;
use App\Http\Helpers\LimitCheckerHelper;
use App\Models\User\RoomBooking;
use App\Services\PaymentGateway\PayPalService;
use App\Traits\MiscellaneousTrait;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Redirect;
use Illuminate\Support\Facades\Session;

class PayPalController extends Controller
{
    use MiscellaneousTrait;

    private $payPalService;
    private $adminId;
    private $tenantId;
    public function __construct()
    {
        $this->payPalService = new PayPalService();
        $this->adminId       = false;
        $this->tenantId      = getUser()->id;
    }

    public function bookingProcess(Request $request)
    {
        $countRoomBooking = LimitCheckerHelper::roomBookingCountUser(getUser()->id);
        $roomBookingLimit = LimitCheckerHelper::roomBookingsLimit(getUser()->id);

        if ($roomBookingLimit > $countRoomBooking) {
            $roomBooking = new RoomBookingController();
            // do calculation
            $calculatedData = $roomBooking->calculation($request);
            $title          = 'Room Booking';
            $currencyInfo   = MiscellaneousTrait::getCurrencyInfo();
            $information['subtotal']                 = $calculatedData['subtotal'];
            $information['discount']                 = $calculatedData['discount'];
            $information['total']                    = $calculatedData['total'];
            $information['userTotal']                = $calculatedData['userTotal'];
            $information['tax']                      = $calculatedData['tax'];
            $information['fee']                      = $calculatedData['fee'];
            $information['currency_symbol']          = $currencyInfo->base_currency_symbol;
            $information['currency_symbol_position'] = $currencyInfo->base_currency_symbol_position;
            $information['currency_text']            = $currencyInfo->base_currency_text;
            $information['currency_text_position']   = $currencyInfo->base_currency_text_position;
            $information['method']                   = 'PayPal';
            $information['type']                     = 'online';
            // store the room booking information in database
            $booking_details = $roomBooking->storeData($request, $information);
            
            // changing the currency before redirect to PayPal
            if ($currencyInfo->base_currency_text !== 'USD') {
                $rate = floatval($currencyInfo->base_currency_rate);
                $convertedTotal = $calculatedData['total'] / $rate;
            }
            $paypalTotal = $currencyInfo->base_currency_text === 'USD' ? $calculatedData['total'] : $convertedTotal;
            
            $notify_url = route('room_booking.paypal.notify', getParam());
            $cancel_url = route('room_booking.cancel', getParam());
            Session::put('bookingId', $booking_details->id); // db row number
            $response = $this->paymentProcess($request, $paypalTotal, $title, $notify_url, $cancel_url, $currencyInfo);
            if (!empty($response) && isset($response['id']) && $response['id'] != null) {
                foreach ($response['links'] as $link) {
                    if ($link['rel'] == 'approve') {
                        return redirect()->away($link['href']);
                    }
                }
            }
            return redirect($cancel_url)->with('error', 'Something went wrong.');
        } else {
            return redirect()->back()->with('error', __('Please Contact Support'));
        }
    }

    public function paymentProcess(Request $request, $amount, $title, $successUrl, $cancelUrl, $be)
    {
        try {
            $this->payPalService->setCredentials($this->adminId, $this->tenantId);
            $amount   = $this->payPalService->convertAmount($be, $amount);
            $response = $this->payPalService->createOrder($amount, $successUrl, $cancelUrl);
            return $response;
        } catch (\Exception $e) {
            return redirect($cancelUrl)->with('error', $e->getMessage());
        }
    }

    public function notify(Request $request)
    {
        

        $payment_success_url = route('room_booking.complete', getParam());
        $payment_cancel_url  = route('room_booking.cancel', getParam());
        try {
            $this->payPalService->setCredentials($this->adminId, $this->tenantId);
            $response = $this->payPalService->captureOrder($request['token']);

            if (! isset($response['error']) && isset($response['status']) && $response['status'] == 'COMPLETED') {
                // update the payment status for room booking in database
                $bookingId   = Session::get('bookingId');
                $bookingInfo = RoomBooking::findOrFail($bookingId);
                $bookingInfo->update(['payment_status' => 1]);
                $roomBooking = new RoomBookingController();
                // generate an invoice in pdf format
                $invoice = $roomBooking->generateInvoice($bookingInfo);
                
                // update the invoice field information in database
                $bookingInfo->update(['invoice' => $invoice]);
          
                // send a mail to the customer with an invoice
                $roomBooking->sendMail($bookingInfo);
               
                // remove all session data
                Session::forget('bookingId');
                return redirect($payment_success_url);
            } else {
                return redirect($payment_cancel_url);
            }
        } catch (\Exception $e) {
            dd($e);
            return redirect($payment_cancel_url)->with('error', $e->getMessage());
        }
    }
}
