<?php

namespace App\Http\Controllers\FrontEnd\Package;

use App\Http\Controllers\Controller;
use App\Http\Helpers\LimitCheckerHelper;
use App\Models\User\PackageBooking;
use App\Services\PaymentGateway\PayPalService;
use App\Traits\MiscellaneousTrait;
use Illuminate\Http\Request;

class PayPalController extends Controller
{
    use MiscellaneousTrait;

    private $payPalService;
    private $adminId;
    private $tenantId;

    public function __construct()
    {
        $this->payPalService = new PayPalService();
        $this->adminId       = false;
        $this->tenantId      = getUser()->id;
    }

    public function bookingProcess(Request $request)
    {
        $countPackageBooking = LimitCheckerHelper::packageBookingCountUser(getUser()->id);
        $packageBookingLimit = LimitCheckerHelper::packageBookingsLimit(getUser()->id);
        
        if ($packageBookingLimit > $countPackageBooking) {
            $packageBooking = new PackageBookingController();
            // do calculation
            $calculatedData = $packageBooking->calculation($request);
            $title          = 'Package Booking';
            $currencyInfo   = MiscellaneousTrait::getCurrencyInfo();

            $information['subtotal']                 = $calculatedData['subtotal'];
            $information['discount']                 = $calculatedData['discount'];
            $information['total']                    = $calculatedData['total'];
            $information['userTotal']                = $calculatedData['userTotal'];
            $information['tax']                      = $calculatedData['tax'];
            $information['fee']                      = $calculatedData['fee'];
            $information['currency_symbol']          = $currencyInfo->base_currency_symbol;
            $information['currency_symbol_position'] = $currencyInfo->base_currency_symbol_position;
            $information['currency_text']            = $currencyInfo->base_currency_text;
            $information['currency_text_position']   = $currencyInfo->base_currency_text_position;
            $information['method']                   = 'PayPal';
            $information['type']                     = 'online';
            $rate                                    = $currencyInfo->base_currency_rate;
            // store the package booking information in database
            $booking_details = $packageBooking->storeData($request, $information);
            $request->session()->put('bookingId', $booking_details->id); // db row number

            if ($currencyInfo->base_currency_text !== 'USD') {
                $rate = floatval($currencyInfo->base_currency_rate);
                $convertedTotal = $calculatedData['total'] / $rate;
            }
            $paypalTotal = $currencyInfo->base_currency_text === 'USD' ? $calculatedData['total'] : $convertedTotal;
            $notify_url = route('package_booking.paypal.notify', getParam());
            $cancel_url = route('package_booking.cancel', getParam());
            $response = $this->paymentProcess($request, $paypalTotal, $title, $notify_url, $cancel_url, $currencyInfo);
          
            if (!empty($response) && isset($response['id']) && $response['id'] != null) {
                foreach ($response['links'] as $link) {
                    if ($link['rel'] == 'approve') {
                        return redirect()->away($link['href']);
                    }
                }
            }
            return redirect($cancel_url)->with('error', 'Something went wrong.');
        } else {
            return redirect()->back()->with('error', __('Please Contact Support.'));
        }
    }
    public function paymentProcess(Request $request, $amount, $title, $successUrl, $cancelUrl, $be)
    {
        try {
            $this->payPalService->setCredentials($this->adminId, $this->tenantId);
            $amount   = $this->payPalService->convertAmount($be, $amount);
            $response = $this->payPalService->createOrder($amount, $successUrl, $cancelUrl);
            return $response;
        } catch (\Exception $e) {
            return redirect($cancelUrl)->with('error', $e->getMessage());
        }
    }

    public function notify(Request $request)
    {
        $payment_success_url = route('package_booking.complete', getParam());
        $payment_cancel_url  = route('package_booking.cancel', getParam());
        try {
            $this->payPalService->setCredentials($this->adminId, $this->tenantId);
            $response = $this->payPalService->captureOrder($request['token']);
            if (! isset($response['error']) && isset($response['status']) && $response['status'] == 'COMPLETED') {
                $bookingId = $request->session()->get('bookingId');
                // update the payment status for package booking in database
                $bookingInfo = PackageBooking::findOrFail($bookingId);
                $bookingInfo->update(['payment_status' => 1]);
                $packageBooking = new PackageBookingController();
                // generate an invoice in pdf format
                $invoice = $packageBooking->generateInvoice($bookingInfo);
                // update the invoice field information in database
                $bookingInfo->update(['invoice' => $invoice]);
                // send a mail to the customer with an invoice
                $packageBooking->sendMail($bookingInfo);
                // remove all session data
                $request->session()->forget('bookingId');
                return redirect($payment_success_url);
            } else {
                return redirect($payment_cancel_url);
            }
        } catch (\Exception $e) {
            return redirect($payment_cancel_url)->with('error', $e->getMessage());
        }
    }
}
