<?php

namespace App\Http\Controllers\FrontEnd\Package;

use App\Http\Controllers\Controller;
use App\Models\User\Coupon;
use App\Models\User\OfflineGateway;
use App\Models\User\Package;
use App\Models\User\PackageBooking;
use App\Models\User\PackageCategory;
use App\Models\User\PackageContent;
use App\Models\User\PackageLocation;
use App\Models\User\PackagePlan;
use App\Models\User\PackageReview;
use App\Models\User\PaymentGateway;
use App\Traits\MiscellaneousTrait;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

class PackageController extends Controller
{
    use MiscellaneousTrait;

    public function packages($domain, Request $request)
    {

        $user                          = getuser();
        $queryResult['breadcrumbInfo'] = MiscellaneousTrait::getBreadcrumb();
        $language                      = MiscellaneousTrait::getLanguage();
        $queryResult['pageHeading']    = $this->getUserPageHeading($language, $user->id);

        $queryResult['packageRating'] = DB::table('user_basic_settings')->select('package_rating_status')->first();

        $queryResult['categories'] = PackageCategory::where('language_id', $language->id)->where('user_id', $user->id)->where('status', 1)->orderBy('serial_number', 'ASC')->get();

        $queryResult['maxPrice'] = Package::max('package_price');

        $queryResult['minPrice']   = Package::min('package_price');
        $queryResult['maxPersons'] = Package::max('max_persons');
        $queryResult['maxDays']    = Package::max('number_of_days');

        $package_name = $sort_value = $location_name = $min_price = $max_price = null;

        $category = $request->category;
        if ($request->filled('packageName')) {
            $package_name = $request->packageName;
        }
        if ($request->filled('personsValue')) {
            $persons_value = $request->personsValue;
        } else {
            $persons_value = 0;
        }
        if ($request->filled('daysValue')) {
            $days_value = $request->daysValue;
        } else {
            $days_value = 0;
        }
        $ratings = $request->ratings ?? null;
   
        
        $sort_value    = $request->sortValue;
        $true          = true;
        $location_name = $request->locationName;
        if ($request->filled('minPrice') && $request->filled('maxPrice')) {
            $min_price = $request->minPrice;
            $max_price = $request->maxPrice;
        }
        $packageIds = [];
        if (!empty($location_name)) {
            $locations = PackageLocation::select('package_id')->where('user_id', $user->id)->where('name', 'LIKE', '%' . $location_name . '%')->get();
            foreach ($locations as $key => $location) {
                if (! in_array($location->package_id, $packageIds)) {
                    $packageIds[] = $location->package_id;
                }
            }
        }
     
        
        $packageInfos = Package::join('user_package_contents', 'user_packages.id', '=', 'user_package_contents.package_id')
            ->where('user_package_contents.language_id', '=', $language->id)
            ->where('user_packages.user_id', '=', $user->id)
            ->when($category, function ($query)use($category, $language, $user) {
                $cat = PackageCategory::where('language_id', $language->id)
                    ->where('user_id', $user->id)
                    ->where('status', 1)
                    ->where('slug', $category)
                    ->first();
                if (!is_null($cat)) {
                    return $query->where('user_packages.package_category_index', $cat->indx);
                }
            })
            ->when($package_name, function ($query, $package_name) {
                return $query->where('title', 'like', '%' . $package_name . '%');
            })->when(($min_price && $max_price), function ($query) use ($min_price, $max_price) {
                return $query->where('package_price', '>=', $min_price)->where('package_price', '<=', $max_price);
            })->when($days_value, function ($query, $days_value) {
                return $query->where('number_of_days', '<=', $days_value);
            })->when($persons_value, function ($query, $persons_value) {
                return $query->where('max_persons', '>=', $persons_value);
            })->when($location_name, function ($query) use ($packageIds) {
                return $query->whereIn('user_packages.id', $packageIds);
            })->when($ratings,function ($query, $ratings) {
                return $query->where('avg_rating', '>=', $ratings);
            });
            
        if ($sort_value == 'new-packages') {
            $packageInfos->orderBy('user_packages.created_at', 'desc');
        } else if ($sort_value == 'old-packages') {
            $packageInfos->orderBy('user_packages.created_at', 'asc');
        } else if ($sort_value == 'price-asc') {
            $packageInfos->orderBy('user_packages.package_price', 'asc');
        } else if ($sort_value == 'price-desc') {
            $packageInfos->orderBy('user_packages.package_price', 'desc');
        } else if ($sort_value == 'max-persons-asc') {
            $packageInfos->orderBy('user_packages.max_persons', 'asc');
        } else if ($sort_value == 'max-persons-desc') {
            $packageInfos->orderBy('user_packages.max_persons', 'desc');
        } else if ($sort_value == 'days-asc') {
            $packageInfos->orderBy('user_packages.number_of_days', 'asc');
        } else if ($sort_value == 'days-desc') {
            $packageInfos->orderBy('user_packages.number_of_days', 'desc');
        } else if ($request->filled('number_of_days')) {
            $packageInfos->orderBy('user_packages.number_of_days', 'desc');
        } else {
            $packageInfos->orderByDesc('user_packages.id');
        }

        $queryResult['packageInfos'] = $packageInfos->paginate(8);
        $queryResult['currencyInfo'] = MiscellaneousTrait::getCurrencyInfo();

        if ($request->ajax() && $request->sidebar_search == 1) {
            return view('user-front.common.package.list_view', $queryResult)->render();
        }
        $queryResult['seo'] = $this->getUserSeoData($language, $user->id);
        return view('user-front.common.package.packages', $queryResult);
    }

    public function packageDetails($domain, $slug)
    {
        $user                          = getuser();
        $queryResult['breadcrumbInfo'] = MiscellaneousTrait::getBreadcrumb();
        $language                      = MiscellaneousTrait::getLanguage();

        $queryResult['packageRating'] = DB::table('user_basic_settings')->where('user_id',$user->id)->select('package_rating_status')->first();

        $details = PackageContent::with('package')
            ->where('language_id', $language->id)
            ->where('slug', $slug)
            ->where('user_id', $user->id)
            ->firstOrFail();

        $id = $details->package_id;
        $queryResult['details'] = $details;

        $queryResult['plans'] = PackagePlan::where('language_id', $language->id)
            ->where('package_id', $id)->where('user_id', $user->id)
            ->get();

        $queryResult['locations'] = PackageLocation::where('language_id', $language->id)
            ->where('package_id', $id)->where('user_id', $user->id)
            ->get();

        $queryResult['reviews'] = PackageReview::where('package_id', $id)->orderBy('id', 'DESC')->where('user_id', $user->id)->get();

        $queryResult['status'] = DB::table('user_basic_settings')
            ->select('package_rating_status', 'package_guest_checkout_status')
            ->where('uniqid', '=', 12345)
            ->first();
        $queryResult['onlineGateways'] = PaymentGateway::where('status', 1)->where('user_id', $user->id)->get();


        $queryResult['offlineGateways'] = OfflineGateway::orderBy('serial_number', 'asc')->where('user_id', $user->id)->get()->map(function ($gateway) {
            return [
                'id'                => $gateway->id,
                'name'              => $gateway->name,
                'short_description' => $gateway->short_description,
                'instructions'      => replaceBaseUrl($gateway->instructions, 'summernote'),
                'is_receipt'        => $gateway->is_receipt,
                'serial_number'     => $gateway->serial_number,
            ];
        });

        $queryResult['currencyInfo'] = MiscellaneousTrait::getCurrencyInfo();



        $queryResult['latestPackages'] = PackageContent::with('package')
            ->where('language_id', $language->id)
            ->where('package_id', '<>', $details->package_id)
            ->whereHas('package', function ($query) use ($details) {
                $query->where('package_category_index', $details->package->package_category_index);
            })
            ->orderBy('package_id', 'desc')
            ->limit(3)
            ->get();


        $queryResult['avgRating'] = PackageReview::where('package_id', $id)->where('user_id', $user->id)->avg('rating');

        $stripe = PaymentGateway::query()
            ->where('keyword', 'stripe')
            ->where('user_id', $user->id)
            ->first();

        if (! is_null($stripe->information)) {
            $stripe                    = json_decode($stripe->information, true);
            $queryResult['stripe_key'] = $stripe['key'];
        }

        $queryResult['anet'] = PaymentGateway::query()
            ->where('keyword', 'authorize.net')
            ->where('user_id', $user->id)
            ->first();
        $queryResult['seo'] = $this->getUserSeoData($language, $user->id);

        return view('user-front.common.package.package_details', $queryResult);
    }

    public function applyCoupon(Request $request)
    {
        $user = getUser();
        $keywords = get_keywords($user->id);
        try {
            $coupon = Coupon::where('code', $request->coupon)->firstOrFail();

            $startDate = Carbon::parse($coupon->start_date);
            $endDate   = Carbon::parse($coupon->end_date);
            $todayDate = Carbon::now();

            // check coupon is valid or not
            if ($todayDate->between($startDate, $endDate) == false) {
                return response()->json(['error' => $keywords['Sorry, coupon has been expired!'] ?? __('Sorry, coupon has been expired!')]);
            }

            // check coupon is valid or not for this package
            $packageId  = $request->packageId;
            $packageIds = empty($coupon->packages) ? '' : json_decode($coupon->packages);

            if (! empty($packageIds) && ! in_array($packageId, $packageIds)) {
                return response()->json(['error' =>  $keywords['You can not apply this coupon for this package!'] ?? __('You can not apply this coupon for this package!')]);
            }

            $request->session()->put('couponCode', $request->coupon);

            $initTotalRent = str_replace(',', '', $request->initTotal);

            if ($coupon->type == 'fixed') {
                $total = floatval($initTotalRent) - floatval($coupon->value);

                return response()->json([
                    'success'  =>  $keywords['Coupon applied successfully.'] ?? __('Coupon applied successfully.'),
                    'discount' => $coupon->value,
                    'total'    => $total,
                ]);
            } else {
                $initTotalRent = floatval($initTotalRent);
                $couponVal     = floatval($coupon->value);

                $discount = $initTotalRent * ($couponVal / 100);
                $total    = $initTotalRent - $discount;

                return response()->json([
                    'success'  => $keywords['Coupon applied successfully.'] ?? __('Coupon applied successfully.'),
                    'discount' => $discount,
                    'total'    => $total,
                ]);
            }
        } catch (ModelNotFoundException $e) {
            return response()->json(['error' => $keywords['Coupon is not valid!'] ?? __('Coupon is not valid!')]);
        }
    }

    public function storeReview(Request $request, $domain, $id)
    {

        $user = getUser();
        $keywords = get_keywords($user->id);

        $booking = PackageBooking::where('user_id', Auth::guard('customer')->user()->user_id)->where('package_id', $id)->where('payment_status', 1)->count();
        if ($booking == 0) {
            $request->session()->flash('error', $keywords['You had not purchased this package yet.'] ?? __("You had not purchased this package yet."));
            return back();
        }

        $rules = [
            'rating' => 'required|numeric',
            'comment' => 'nullable',

        ];

        $message = [
            'rating.required' =>  $keywords['The star rating field is required.'] ?? __('The star rating field is required.'),
            'comment.required' => $keywords['The comment field is required.'] ?? __('The comment field is required.'),
        ];

        $validator = Validator::make($request->all(), $rules, $message);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        $customer = Auth::guard('customer')->user();

        $review = PackageReview::where('user_id', $customer->user_id)->where('package_id', $id)
            ->first();

        /**
         * if, package review of auth user does not exist then create a new one.
         * otherwise, update the existing review of that auth user.
         */
        if ($review == null) {
            PackageReview::create($request->except('user_id', 'package_id') + [
                'user_id'     => $customer->user_id,
                'customer_id' => $customer->id,
                'package_id'  => $id,
            ]);

            // now, store the average rating of this package
            $package = Package::where('user_id', getUser()->id)->findOrFail($id);

            $package->update(['avg_rating' => $request->rating]);
        } else {
            $review->update($request->all());

            // now, get the average rating of this package
            $packageReviews = PackageReview::where('package_id', $id)->get();

            $totalRating = 0;

            foreach ($packageReviews as $packageReview) {
                $totalRating += $packageReview->rating;
            }

            $avgRating = $totalRating / $packageReviews->count();
            // finally, store the average rating of this package
            $package = Package::where('user_id', getUser()->id)->findOrFail($id);

            $package->update(['avg_rating' => $avgRating]);
        }

        $request->session()->flash('success', $keywords['Review saved successfully!'] ?? __('Review saved successfully!'));

        return redirect()->back();
    }
}
