<?php

namespace App\Http\Controllers\FrontEnd\Package;

use App\Constants\Constant;
use App\Models\User\Package;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use App\Http\Helpers\MegaMailer;
use App\Models\User\PackageCoupon;
use App\Traits\MiscellaneousTrait;
use Illuminate\Support\Facades\DB;
use App\Models\User\PackageBooking;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use App\Http\Controllers\FrontEnd\Package\PaytmController;
use App\Http\Controllers\FrontEnd\Package\MollieController;
use App\Http\Controllers\FrontEnd\Package\PayPalController;
use App\Http\Controllers\FrontEnd\Package\StripeController;
use App\Http\Controllers\FrontEnd\Package\OfflineController;
use App\Http\Controllers\FrontEnd\Package\PaystackController;
use App\Http\Controllers\FrontEnd\Package\RazorpayController;
use App\Http\Controllers\FrontEnd\Package\InstamojoController;
use App\Http\Controllers\FrontEnd\Package\FlutterwaveController;
use App\Http\Controllers\FrontEnd\Package\MercadoPagoController;
use App\Http\Controllers\FrontEnd\Package\AuthorizenetController;
use App\Models\User;
use App\Models\User\BasicSetting;
use App\Models\User\Language;
use Illuminate\Support\Facades\Session;
use Barryvdh\DomPDF\Facade\Pdf;

class PackageBookingController extends Controller
{
  use MiscellaneousTrait;

  public function makePackageBooking(Request $request)
  {
    $user = getUser();
    $keywords = get_keywords($user->id);

    // check whether user is logged in or not (start)
    $status = DB::table('user_basic_settings')->select('package_guest_checkout_status')
    ->where('user_id',getUser()->id)
    ->where('uniqid', '=', 12345)
    ->first();

    if (($status->package_guest_checkout_status == 0) && (Auth::guard('customer')->check() == false)) {
      return redirect()->route('customer.login', [getParam(), 'redirectPath' => 'room_details'
      ]);
    }
    // check whether user is logged in or not (end)

    $package = Package::findOrFail($request->package_id);
    $maxPersons = $package->max_persons;

    if (!empty($request->package_id) && !empty($request->visitors)) {
      Session::put('package_booking_id_visitors', 'package_id_' . $request->package_id . '_visitors_' . str_replace(' ', '',  $request->visitors));
    }

    // validation starts
    $rules = [
      'customer_name' => 'required',
      'customer_phone' => 'required',
      'paymentType' => 'required',
      'customer_email' => 'required|email:rfc,dns',
      'visitors' => [
        'required',
        'numeric',
        function ($attribute, $value, $fail) use ($maxPersons) {
          if (!is_null($maxPersons) && $value > $maxPersons) {
            $fail('Number of ' . $attribute . ' exceeded');
          }
        }
      ]
    ];

    $messages = [
            'customer_name.required' => $keywords['The customer name field is required.'] ?? __('The customer name field is required.'),
            'customer_phone.required' => $keywords['The customer phone field is required.'] ?? __('The customer phone field is required.'),
            'customer_email.required' => $keywords['The customer email field is required.'] ?? __('The customer email field is required.'),
            'visitors.required' => $keywords['The visitors email field is required.'] ?? __('The visitors email field is required.'),
            'visitors.numeric' => $keywords['The visitors field must be numeric.'] ?? __('The visitors field must be numeric.'),
           'paymentType.required' => $keywords['The payment type field is required.'] ?? __('The payment type field is required.'),
    ];


    $validator = Validator::make($request->all(), $rules , $messages);

    if ($validator->fails()) {
      return redirect()->back()->withErrors($validator)->withInput();
    }
    // validation ends
    if ($request->paymentType == 'paypal') {
      $paypal = new PayPalController();
      return $paypal->bookingProcess($request);
    } else if ($request->paymentType == 'stripe') {
      $stripe = new StripeController();

      return $stripe->bookingProcess($request);
    } else if ($request->paymentType == 'instamojo') {
      $instamojo = new InstamojoController();

      return $instamojo->bookingProcess($request);
    } else if ($request->paymentType == 'paystack') {
      $paystack = new PaystackController();

      return $paystack->bookingProcess($request);
    } else if ($request->paymentType == 'authorize.net') {
      $authorizenet = new AuthorizenetController();

      return $authorizenet->bookingProcess($request);
    } else if ($request->paymentType == 'razorpay') {
      $razorpay = new RazorpayController();

      return $razorpay->bookingProcess($request);
    } else if ($request->paymentType == 'mollie') {
      $mollie = new MollieController();

      return $mollie->bookingProcess($request);
    } else if ($request->paymentType == 'paytm') {
      $paytm = new PaytmController();

      return $paytm->bookingProcess($request);
    } else if ($request->paymentType == 'mercadopago') {
      $mercadopago = new MercadoPagoController();

      return $mercadopago->bookingProcess($request);
    } else if ($request->paymentType == 'flutterwave') {
      $flutterwave = new FlutterwaveController();

      return $flutterwave->bookingProcess($request);
    } else {
      $offline = new OfflineController();

      return $offline->bookingProcess($request);
    }
  }

  public function calculation(Request $request)
  {

      $packageInfo = Package::find($request->package_id);
      $userBs = BasicSetting::where('user_id', getUser()->id)->first();
      if ($packageInfo->pricing_type == 'per-person') {
        $adminProfit = 0;
        $userTotal = floatval($packageInfo->package_price) * intval($request->visitors);
      } else {
        $userTotal = floatval($packageInfo->package_price);
        $adminProfit = 0;
      }
      
     $package_fee = $userBs->package_fee_status == 1 ?  $userBs->package_fee : 0.00;

      if ($request->session()->has('couponCode')) {
        $coupon_code = $request->session()->get('couponCode');
        $coupon = PackageCoupon::where('code', $coupon_code)->where('user_id',getUser()->id)->first();

        if (!is_null($coupon)) {
          $couponVal = floatval($coupon->value);

          if ($coupon->type == 'fixed') {
            $subtotal = $userTotal - $couponVal;
            $package_tax = $userBs->package_tax_status == 1 ? $this->tax(($subtotal), $userBs->package_tax) : 0.00;
            $total = ($subtotal + $package_tax + $package_fee);
            $calculatedData = array(
              'subtotal' => $subtotal,
              'discount' => $couponVal,
              'tax' => $package_tax,
              'fee' => $package_fee,
              'total' => $total,
              'userTotal' => $userTotal,
            );
          } else {
            
            $discount = $userTotal * ($couponVal / 100);
            $subtotal = $userTotal - $discount;
            $package_tax = $userBs->package_tax_status == 1 ? $this->tax(($subtotal), $userBs->package_tax) : 0.00;
            $total = ($subtotal + $package_fee + $package_tax);
            $adminProfit = 0;

            $calculatedData = array(
              'subtotal' => $subtotal,
              'discount' => $discount,
              'tax' => $package_tax,
              'fee' => $package_fee,
              'total' => $total,
              'userTotal' => $userTotal,
            );
          }
        } else {
        $discount = 0.00;
        $subtotal = $userTotal - $discount;
        $package_tax = $userBs->package_tax_status == 1 ? $this->tax(($subtotal), $userBs->package_tax) : 0;
        $total = ($subtotal + $package_fee + $package_tax);
        $calculatedData = array(
          'subtotal' => $subtotal,
          'discount' => $discount,
          'tax' => $package_tax,
          'fee' => $package_fee,
          'total' => $total,
          'adminProfit' =>  $adminProfit,
          'userTotal' => $userTotal,
        );
        }
      } else {
        $discount = 0.00;
        $subtotal = $userTotal - $discount;
        $package_tax = $userBs->package_tax_status == 1 ? $this->tax(($subtotal), $userBs->package_tax) : 0.00;
        $total = ($subtotal + $package_fee + $package_tax);
        $calculatedData = array(
          'subtotal' => $subtotal,
          'discount' => $discount,
          'tax' => $package_tax,
          'fee' => $package_fee,
          'total' => $total,
          'adminProfit' =>  $adminProfit,
          'userTotal' => $userTotal,
        );
      }

      $request->session()->forget('couponCode');
      return $calculatedData;

  }

  public function storeData(Request $request, $information)
  {

    $user = User::with(['memberships' => function ($q) {
      $q->where('status', 1)
        ->whereDate('start_date', '<=', Carbon::today())
        ->whereDate('expire_date', '>=', Carbon::today());
    }])->where('id', getUser()->id)->first();
    if (!is_null($user) && $user->memberships->isNotEmpty()) {
      $membership_id =  $user->memberships->first()->id;
    }



    $user = getUser();
    $booking_details = PackageBooking::create([
      'booking_number' => time(),
      'user_id' => $user->id,
      'user_membership_id' => isset($membership_id) ? $membership_id : null,
      'customer_id' => Auth::guard('customer')->check() == true ? Auth::guard('customer')->user()->id : null,
      'customer_name' => $request->customer_name,
      'customer_email' => $request->customer_email,
      'customer_phone' => $request->customer_phone,
      'package_id' => $request->package_id,
      'visitors' => $request->visitors,
      'subtotal' => $information['subtotal'],
      'discount' => $information['discount'],
      'tax' => $information['tax'],
      'fee' => $information['fee'],
      'grand_total' => $information['total'],
      'actual_total' => $information['userTotal'],
      'currency_symbol' => $information['currency_symbol'],
      'currency_symbol_position' => $information['currency_symbol_position'],
      'currency_text' => $information['currency_text'],
      'currency_text_position' => $information['currency_text_position'],
      'payment_method' => $information['method'],
      'gateway_type' => $information['type'],
      'attachment' => $request->hasFile('attachment') ? $information['attachment'] : null
    ]);

    return $booking_details;
  }

  public function generateInvoice($bookingInfo)
  {
    $fileName = $bookingInfo->booking_number . '.pdf';
    $directory = public_path(Constant::WEBSITE_PACKAGE_BOOKING_INVOICE . '/' );

    if (!file_exists($directory)) {
      mkdir($directory, 0777, true);
    }

    $fileLocated = $directory . $fileName;
    $user = getUser();
    $language = Language::query()->where('is_default', '=', 1)->where('user_id', '=', $user->id)->first();
    $keywords = get_keywords($user->id);

    Pdf::setOptions([
      'isHtml5ParserEnabled' => true,
      'isRemoteEnabled' => true,
      'logOutputFile' => storage_path('logs/log.htm'),
      'tempDir' => storage_path('logs/'),
    ])->loadView('user-front.common.pdf.package_booking', compact('bookingInfo','language', 'keywords','user'))->save($fileLocated);

    return $fileName;
  }

  public function sendMail($bookingInfo)
  {

    $info = DB::table('basic_extendeds')->first();
    $infoCus = DB::table('user_basic_settings')->first();

    // get the package name according to language
    $language = MiscellaneousTrait::getLanguage();

    $packageInfo = $bookingInfo->tourPackage()->first();

    $packageContentInfo = $packageInfo->packageContent
      ->where('language_id', $language->id)
      ->first();

    $packageName = $packageContentInfo->title;

    // get the package price and currency information
    $currencyInfo = MiscellaneousTrait::getCurrencyInfo();

    $price = $bookingInfo->grand_total;

    $packagePrice = ($currencyInfo->base_currency_text_position == 'left' ? $currencyInfo->base_currency_text . ' ' : '') . $price . ($currencyInfo->base_currency_text_position == 'right' ? ' ' . $currencyInfo->base_currency_text : '');

    $couponDiscount = $bookingInfo->discount ?? 0;
    
    $couponDiscount = ($currencyInfo->base_currency_text_position == 'left' ? $currencyInfo->base_currency_text . ' ' : '') . $couponDiscount . ($currencyInfo->base_currency_text_position == 'right' ? ' ' . $currencyInfo->base_currency_text : '');


    $mailer = new MegaMailer();
    $data = [
      'toMail' => $bookingInfo->customer_email,
      'toName' =>  $bookingInfo->customer_name,
      'customer_name' => $bookingInfo->customer_name,
      'booking_number' => $bookingInfo->booking_number,
      'booking_date' =>  date_format($bookingInfo->created_at, 'F d, Y'),
      'website_title' => $infoCus->website_title,
      'package_name' => $packageName,
      'package_price' => $packagePrice,
      'number_of_visitors' => $bookingInfo->visitors,
      'check_in_date' => $bookingInfo->arrival_date,
      'discount' => $couponDiscount,
      'check_out_date' => $bookingInfo->departure_date,
      'number_of_guests' => $bookingInfo->guests,
      'templateType' => $bookingInfo->gateway_type == 'Offline' ? 'package_booking_for_offline_gateway' : 'package_booking_for_online_gateway',
      'invoice' => public_path(Constant::WEBSITE_PACKAGE_BOOKING_INVOICE . '/' . $bookingInfo->invoice),
      'type' => 'packagebooking',
    ];
    $mailer->mailFromUser($data, getUser());

  }

  public function applyCoupon(Request $request)
  {
        $user = getUser();
        $keywords = get_keywords($user->id);

    try {

      $coupon = PackageCoupon::where('code', $request->coupon)->where('user_id',getUser()->id)->firstOrFail();
      $startDate = Carbon::parse($coupon->start_date);
      $endDate = Carbon::parse($coupon->end_date);
      $todayDate = Carbon::now();

      // check coupon is valid or not
      if ($todayDate->between($startDate, $endDate) == false) {
        return response()->json(['error' => $keywords['Sorry, coupon has been expired!'] ?? __('Sorry, coupon has been expired!')]);
      }

      // check coupon is valid or not for this package
      $packageId = $request->packageId;
      $packageIds = empty($coupon->packages) ? '' : json_decode($coupon->packages);

      if (!empty($packageIds) && !in_array($packageId, $packageIds)) {
        return response()->json(['error' => $keywords['You can not apply this coupon for this package!'] ?? __('You can not apply this coupon for this package!') ]);
      }

      $request->session()->put('couponCode', $request->coupon);

      $initTotalRent = str_replace(',', '', $request->initTotal);

      if ($coupon->type == 'fixed') {
        $total = floatval($initTotalRent) - floatval($coupon->value);

        return response()->json([
          'success' =>  $keywords['Coupon applied successfully.'] ?? __('Coupon applied successfully.'),
          'discount' => $coupon->value,
          'total' => $total,
        ]);
      } else {
        $initTotalRent = floatval($initTotalRent);
        $couponVal = floatval($coupon->value);

        $discount = $initTotalRent * ($couponVal / 100);
        $total = $initTotalRent - $discount;

        return response()->json([
          'success' => $keywords['Coupon applied successfully.'] ?? __('Coupon applied successfully.'),
          'discount' => $discount,
          'total' => $total
        ]);
      }
    } catch (ModelNotFoundException $e) {
      return response()->json(['error' => $keywords['Coupon is not valid!'] ?? __('Coupon is not valid!')]);
    }
  }
  public function removeCoupon(Request $request)
  {
    if (Session::has('couponCode')) {
      Session::forget('couponCode');
    }
    return "success";
  }
  public function complete()
  {
    $user = getuser();
    $queryResult['breadcrumbInfo'] = $this->getUserBreadcrumb($user->id);
    return view('user-front.common.partials.payment_success', $queryResult);
  }

  public function cancel()
  {

    return redirect()->route('packages',getParam())->with('error', 'Sorry, an error has occured!');
  }
}
