<?php

namespace App\Http\Controllers\FrontEnd;

use App\Constants\Constant;
use Carbon\Carbon;
use App\Models\Customer;
use App\Models\User\SEO;
use Illuminate\Support\Str;
use Illuminate\Http\Request;
use App\Http\Helpers\MegaMailer;
use App\Models\User\RoomBooking;
use App\Models\User\BasicSetting;
use App\Traits\MiscellaneousTrait;
use Illuminate\Support\Facades\DB;
use App\Models\User\PackageBooking;
use App\Http\Controllers\Controller;
use App\Http\Helpers\Uploader;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\Session;
use Laravel\Socialite\Facades\Socialite;
use App\Http\Requests\UserProfileRequest;
use Illuminate\Support\Facades\Validator;
use Illuminate\Database\Eloquent\ModelNotFoundException;

class CustomerController extends Controller
{
    public function __construct()
    {
        $user = getUser();
        $bs = BasicSetting::where('user_id', $user->id)->first();
        Config::set('captcha.sitekey', $bs->google_recaptcha_site_key);
        Config::set('captcha.secret', $bs->google_recaptcha_secret_key);


        Config::set('services.google.client_id', $bs->google_client_id);
        Config::set('services.google.client_secret', $bs->google_client_secret);
        Config::set('services.google.redirect', route('user.client.google.callback', getParam()));
    }

    public function login(Request $request, $domain)
    {
        $user = getUser();
        $queryResult['bgImg'] = $this->getUserBreadcrumb($user->id);
        $language = MiscellaneousTrait::getLanguage();
        $queryResult['pageHeading'] = $this->getUserPageHeading($language, $user->id);
        
        $queryResult['seo'] = $this->getUserSeoData($language, $user->id);
        if (!empty(request()->input('redirectPath')) &&(request()->input('redirectPath') == 'room_details')) {
            $url = url()->previous();
        }else if (!empty(request()->input('redirectPath')) &&(request()->input('redirectPath') == 'package_details')) {
            $url = url()->previous();
        }
        if (isset($url)) {
            Session::put('redirectTo', $url);
        }

        return view('user-front.common.customer.auth.login', $queryResult);
    }

    public function loginSubmit(Request $request, $domain)
    {
        // at first, get the url from session which will be redirected after login
        $keywords = get_keywords(getUser()->id);
        if (Session::has('redirectTo')) {
            $redirectURL = Session::get('redirectTo');
        } else {
            $redirectURL = null;
        }
        $rules = [
            'email' => 'required|email',
            'password' => 'required'
        ];
        $messages= [
            'email.required' => $keywords['The email field is required.'] ?? __('The email field is required.'),
            'password.required' => $keywords['The password field is required.'] ?? __('The password field is required.'),
            'email.email' => $keywords['Please enter validated email.'] ?? __('Please enter validated email.'),
        ];
        $settings = BasicSetting::where('user_id', getUser()->id)->first();
        if ($settings->google_recaptcha_status == 1) {
            $rules['g-recaptcha-response'] = 'required|captcha';
        }
        if ($settings->google_recaptcha_status == 1) {
            $messages['g-recaptcha-response.required'] = $keywords['Please verify that you are not a robot.'] ?? __('Please verify that you are not a robot.');
            $messages['g-recaptcha-response.captcha'] = $keywords['Captcha error! try again later or contact site admin.'] ?? __('Captcha error! try again later or contact site admin.');
        }

        $validator = Validator::make($request->all(), $rules, $messages);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        // get the email and password which has provided by the user
        $credentials = $request->only('email', 'password', 'user_id');
     
        // login attempt
        if (Auth::guard('customer')->attempt($credentials)) {
            $authUser = Auth::guard('customer')->user();
            // first, check whether the user's email address verified or not
            if ($authUser->email_verified_at == null) {
                Session::flash('error', $keywords['Please, verify your email address.'] ?? __('Please, verify your email address.'));
                // logout auth user as condition not satisfied
                Auth::guard('customer')->logout();
                return redirect()->back();
            }

            // second, check whether the user's account is active or not
            if ($authUser->status == 0) {
                Session::flash('error', $keywords['Sorry, your account has been deactivated.'] ?? __('Sorry, your account has been deactivated.'));
                // logout auth user as condition not satisfied
                Auth::guard('customer')->logout();
                return redirect()->back();
            }
            
           
            // otherwise, redirect auth user to next url
            if ($redirectURL == null) {
                return redirect()->route('customer.dashboard', getParam());
            } else {
                // before, redirect to next url forget the session value
                Session::forget('redirectTo');
                return redirect($redirectURL);
            }
        } else {
            Session::flash('error', $keywords['The provided credentials do not match our records!'] ?? __('The provided credentials do not match our records!'));
            return redirect()->back();
        }
    }

    public function forgetPassword($domain)
    {
        $user = getUser();
        $language = MiscellaneousTrait::getLanguage();
        $queryResult['seoInfo'] = SEO::query()
            ->where('user_id', $user->id)
            ->select('forget_password_meta_keywords', 'forget_password_meta_description')
            ->first();
        $queryResult['pageHeading'] = $this->getUserPageHeading($language, $user->id);
        $queryResult['bgImg'] = $this->getUserBreadcrumb($user->id);
        $queryResult['seo'] = $this->getUserSeoData($language, $user->id);
        return view('user-front.common.customer.auth.forget-password', $queryResult);
    }
    public function sendMail(Request $request)
    {
        $user = getUser();
        $keywords = get_keywords($user->id);
        $err_msg =  $keywords['No record found for'] ?? __('No record found for'); 
        $rules = [
            'email' => [
                'required',
                'email:rfc,dns',
                function ($attribute, $value, $fail) use ($request, $user, $err_msg) {
                    if (Customer::where('email', $request->email)->where('user_id', $user->id)->count() == 0) {
                        $fail( $err_msg .' ' . $request->email);
                    }
                }
            ]
        ];

        $messages = [
            'email.required' => $keywords['The email field is required.'] ?? __('The email field is required.'),
            'email.email' =>   $keywords['Please enter validated email.'] ?? __('Please enter validated email.'),
        ];
        
        $settings = BasicSetting::where('user_id', getUser()->id)->first();
        if ($settings->google_recaptcha_status == 1) {
            $rules['g-recaptcha-response'] = 'required|captcha';
        }
        if ($settings->google_recaptcha_status == 1) {
            $messages['g-recaptcha-response.required'] = $keywords['Please verify that you are not a robot.'] ?? __('Please verify that you are not a robot.');
            $messages['g-recaptcha-response.captcha'] = $keywords['Captcha error! try again later or contact site admin.'] ?? __('Captcha error! try again later or contact site admin.');
        }

        $validator = Validator::make($request->all(), $rules, $messages);
        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        $customer = Customer::where('email', $request->email)->where('user_id', $user->id)->first();

        $pass_token = Str::random(30);
        $input['pass_token'] = $pass_token;
        $customer->update($input);
        $forgetLinkRoute = route('password.reset.form', [getParam(), 'pass_token' => $customer->pass_token]);
        $btn = "<a href='$forgetLinkRoute'>Please click this link to create a new password.</a>";
        $userBs = BasicSetting::where('user_id', $user->id)->select('website_title', 'email', 'from_name')->first();

        $mailer = new MegaMailer();
        $data = [
            'toMail' => $customer->email,
            'toName' => $customer->username,
            'customer_name' => $customer->username,
            'password_reset_link' => $btn,
            'templateType' => 'reset_password',
            'website_title' => $userBs->website_title,
            'type' => 'reset_password'
        ];
        $mailer->mailFromUser($data, $user);

        // store user email in session to use it later
        Session::put('userEmail', $customer->email);

        return redirect()->back()->with('success', $keywords['A mail was send to your email'] ?? __('A mail was send to your email'));
    }

    public function resetPassword($domain)
    {
        $user = getUser();
        $queryResult['bgImg'] = $this->getUserBreadcrumb($user->id);
        $language = MiscellaneousTrait::getLanguage();
        $queryResult['pageHeading'] = $this->getUserPageHeading($language, $user->id);
        
        return view('user-front.common.customer.auth.reset-password', $queryResult);
    }

    public function resetPasswordSubmit(Request $request, $domain)
    {
        $author = getUser();
        $keywords = get_keywords($author->id);
        // get the user email from session
        $emailAddress = Session::get('userEmail');
        $messages = [
            'password.required' => $keywords['The password field is required.'] ?? __('The password field is required.'),
            'password.min' => $keywords['The password field must be at least 8 characters.'] ?? __('The password field must be at least 8 characters.')
        ];

        $request->validate([
            'password' => 'required|min:8',
        ],$messages);
        $Customer = Customer::where('email', $emailAddress)->where('user_id', $author->id)->where('pass_token', $request->pass_token)->first();
        if ($Customer) {
            $Customer->password = bcrypt($request->password);
            $Customer->pass_token = null;
            $Customer->save();
            return back()->with('success', $keywords['New password created successfully'] ?? __('New password created successfully'));
        } else {
            Session::flash('warning', $keywords['This link has expired. Please request it again.'] ?? __('This link has expired. Please request it again.'));
            return back();
        }
    }

    public function signup($domain)
    {
        $user = getUser();
        $language = MiscellaneousTrait::getLanguage();
        $queryResult['pageHeading'] = $this->getUserPageHeading($language, $user->id);
        
        $queryResult['bgImg'] = $this->getUserBreadcrumb($user->id);
        $queryResult['seoInfo'] = SEO::query()
            ->where('user_id', $user->id)
            ->select('sign_up_meta_keywords', 'sign_up_meta_description')
            ->first();
        $queryResult['seo'] = $this->getUserSeoData($language, $user->id);
        return view('user-front.common.customer.auth.signup', $queryResult);
    }

    public function signupSubmit(Request $request, $domain)
    {
        $user = getUser();
        $settings = BasicSetting::where('user_id',$user->id)->first();
        $keywords = get_keywords($user->id);
        $rules = [
            'username' => [
                'required',
                'max:255',
                function ($attribute, $value, $fail) use ($user) {
                    if (Customer::where('username', $value)->where('user_id', $user->id)->count() > 0) {
                        $fail( $keywords['Username has already been taken'] ?? __('Username has already been taken'));
                    }
                }
            ],
            'email' => ['required', 'email', 'max:255', function ($attribute, $value, $fail) use ($user,$keywords) {
                if (Customer::where('email', $value)->where('user_id', $user->id)->count() > 0) {
                    $fail( $keywords['Email has already been taken'] ?? __('Email has already been taken'));
                }
            }],
            'password' => 'required|confirmed',
            'password_confirmation' => 'required'
        ];

        $messages = [
            'username.required' =>  $keywords['The username field is required.'] ?? __('The username field is required.'),
            'username.max' => $keywords['The username max length 255.'] ?? __('The username max length 255.'),
            'email.required' =>  $keywords['The email field is required.'] ?? __('The email field is required.'),
            'email.email' => $keywords['Please enter validated email.'] ?? __('Please enter validated email.'),
            'password.required' => $keywords['The password field is required.'] ?? __('The password field is required.'),
            'password.confirmed' =>  $keywords['The password field confirmation does not match.'] ?? __('The password field confirmation does not match.'),
            'password_confirmation.required' => $keywords['The password confirmation field is required.'] ?? __('The password confirmation field is required.'),
        ];
      

        if ($settings->google_recaptcha_status == 1) {
            $rules['g-recaptcha-response'] = 'required|captcha';
        }
        if ($settings->google_recaptcha_status == 1) {
            $messages['g-recaptcha-response.required'] = $keywords['Please verify that you are not a robot.'] ?? __('Please verify that you are not a robot.');
            $messages['g-recaptcha-response.captcha'] = $keywords['Captcha error! try again later or contact site admin.'] ?? __('Captcha error! try again later or contact site admin.');
        }

        $validator = Validator::make($request->all(), $rules, $messages);
        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }
        $customer = new Customer;
        $customer->username = $request->username;
        $customer->email = $request->email;
        $customer->user_id = $user->id;
        $customer->password = Hash::make($request->password);

        // first, generate a random string
        $randStr = Str::random(20);
        // second, generate a token
        $token = md5($randStr . $request->username . $request->email);
        $customer->verification_token = $token;
        $customer->save();

        $bs = DB::table('user_basic_settings')
            ->select('website_title')->where('user_id', getUser()->id)
            ->first();

        $link = '<a href=' . route('customer.signup.verify', ['token' => $token, getParam()]) . '>Click Here</a>';

        $mailer = new MegaMailer();
        $data = [
            'toMail' => $customer->email,
            'toName' => $customer->username,
            'username' => $customer->username,
            'customer_name' => $customer->username,
            'verification_link' => $link,
            'website_title' => $bs->website_title,
            'templateType' => 'verify_email',
            'type' => 'emailVerification'
        ];
        // send a mail to user for verify his/her email address
        $mailer->mailFromUser($data, $user);
        Session::flash('success', $keywords['A verification mail has been sent to your email address.'] ?? __('A verification mail has been sent to your email address.'));
        return redirect()->back();
    }


    public function signupVerify(Request $request, $domain, $token)
    {
        $user = getUser();
        $keywords = get_keywords($user->id);
        try {
            $user = Customer::where('verification_token', $token)->firstOrFail();
            // after verify user email, put "null" in the "verification token"
            $user->update([
                'email_verified_at' => date('Y-m-d H:i:s'),
                'status' => 1,
                'verification_token' => null
            ]);
            Session::flash('success', $keywords['Your email has verified.'] ?? __('Your email has verified.'));
            // after email verification, authenticate this user
            Auth::guard('customer')->login($user);
            return redirect()->route('customer.dashboard', getParam());
        } catch (ModelNotFoundException $e) {
            Session::flash('error', $keywords['Could not verify your email!'] ?? __('Could not verify your email!'));
            return redirect()->route('customer.signup', getParam());
        }
    }

    public function redirectToDashboard($domain)
    {
      
        $user = getUser();
        $language = MiscellaneousTrait::getLanguage();
        $queryResult['pageHeading'] = $this->getUserPageHeading($language, $user->id);
        $queryResult['seo'] = $this->getUserSeoData($language, $user->id);
        $queryResult['bgImg'] = $this->getUserBreadcrumb($user->id);
        $queryResult['authUser'] = Auth::guard('customer')->user();

        $queryResult['totalRoomBooking'] = RoomBooking::where('user_id', $user->id)->where('customer_id', Auth::guard('customer')->user()->id)->count();
        $queryResult['totalPackageBooking'] = PackageBooking::where('user_id', $user->id)->where('customer_id', Auth::guard('customer')->user()->id)->count();
        return view('user-front.common.customer.dashboard', $queryResult);
    }

    public function editProfile($domain)
    {
        $user = getUser();
        $language = MiscellaneousTrait::getLanguage();
        $queryResult['pageHeading'] = $this->getUserPageHeading($language, $user->id);
        $queryResult['seo'] = $this->getUserSeoData($language, $user->id);
        
        $queryResult['bgImg'] = $this->getUserBreadcrumb($user->id);
        $queryResult['authUser'] = Auth::guard('customer')->user();
        return view('user-front.common.customer.edit-profile', $queryResult);
    }

    public function updateProfile(UserProfileRequest $request, $domain)
    {
        $user = getUser();
        $keywords = get_keywords($user->id);
        $customer = Auth::guard('customer')->user();
        $image = $customer->image;
        if ($request->hasFile('image')) {
            Uploader::remove(public_path(Constant::WEBSITE_CUSTOMER), $customer->image);
            $image = Uploader::upload_picture(public_path(Constant::WEBSITE_CUSTOMER), $request->file('image'));
        }

        $customer->update($request->except('image', 'edit_profile_status') + [
            'image' => $image,
            'edit_profile_status' => 1
        ]);
        Session::flash('success', $keywords['Your profile updated successfully.'] ?? __('Your profile updated successfully.'));
        return redirect()->back();
    }

    public function myRoomBoking($domain)
    {
        $user = getUser();
        $language = MiscellaneousTrait::getLanguage();
        $queryResult['pageHeading'] = $this->getUserPageHeading($language, $user->id);
        $queryResult['seo'] = $this->getUserSeoData($language, $user->id);
        
        
        $authUser = Auth::guard('customer')->user();
        $queryResult['bgImg'] = $this->getUserBreadcrumb($user->id);
        $queryResult['roomBookingInfos'] = RoomBooking::where('user_id',$user->id)->where('customer_id', $authUser->id)->orderBy('id', 'DESC')->get();
        $queryResult['langInfo'] = MiscellaneousTrait::getLanguage();
        return view('user-front.common.user.my_room_bookings', $queryResult);
    }
    public function roomBookingDetails($domain, $id)
    {
        $user = getUser();
        $language = MiscellaneousTrait::getLanguage();
        $queryResult['bgImg'] = $this->getUserBreadcrumb($user->id);
        $queryResult['seo'] = $this->getUserSeoData($language, $user->id);
        $roomBooking = RoomBooking::findOrFail($id);
        if ($roomBooking->customer_id != Auth::guard('customer')->user()->id) {
            return back();
        }
        $queryResult['details'] = $roomBooking;
        $queryResult['userInfo'] = $roomBooking->roomBookedByCustomer()->firstOrFail();
        return view('user-front.common.user.room_booking_details', $queryResult);
    }
    public function packageBookings($domain)
    {
        $user = getUser();
        $language = MiscellaneousTrait::getLanguage();
        $queryResult['pageHeading'] = $this->getUserPageHeading($language, $user->id);
        $queryResult['seo'] = $this->getUserSeoData($language, $user->id);
        
        $queryResult['bgImg'] = $this->getUserBreadcrumb($user->id);
        $authUser = Auth::guard('customer')->user();
        $queryResult['packageBookingInfos'] = PackageBooking::where('user_id',$user->id)->where('customer_id', $authUser->id)->orderBy('id', 'DESC')->get();
        $queryResult['langInfo'] = MiscellaneousTrait::getLanguage();
        return view('user-front.common.user.package_bookings', $queryResult);
    }
    public function packageBookingDetails($domain, $id)
    {
        $user = getUser();
        $language = MiscellaneousTrait::getLanguage();
        $queryResult['seo'] = $this->getUserSeoData($language, $user->id);
        
        $queryResult['bgImg'] = $this->getUserBreadcrumb($user->id);
        $packageBooking = PackageBooking::findOrFail($id);
        $queryResult['details'] = $packageBooking;
        $queryResult['userInfo'] = $packageBooking->packageBookedByCustomer()->firstOrFail();
        return view('user-front.common.user.package_booking_details', $queryResult);
    }

  

    public function changePassword($domain)
    {
        $user = getUser();
        $language = MiscellaneousTrait::getLanguage();
        $queryResult['seo'] = $this->getUserSeoData($language, $user->id);
        $queryResult['pageHeading'] = $this->getUserPageHeading($language, $user->id);
        $queryResult['bgImg'] = $this->getUserBreadcrumb($user->id);
        return view('user-front.common.customer.change-password', $queryResult );
    }

    public function updatePassword(Request $request, $domain)
    {
        $user = getUser();
        $keywords = get_keywords($user->id);
        $err_msg = $keywords['Your password was not updated, since the provided current password does not match.'] ?? __('Your password was not updated, since the provided current password does not match.');

        $rules = [
            'current_password' => [
                'required',
                function ($attribute, $value, $fail) use ($err_msg) {
                    if (!Hash::check($value, Auth::guard('customer')->user()->password)) {
                        $fail($err_msg);
                    }
                }
            ],
            'new_password' => 'required|confirmed',
            'new_password_confirmation' => 'required'
        ];

        $messages = [
            'current_password.required' => $keywords['The password field is required.'] ?? __('The password field is required.'),
            'new_password.required' => $keywords['The password field is required.'] ?? __('The password field is required.'),
            'new_password.confirmed' =>  $keywords['The password field confirmation does not match.'] ?? __('The password field confirmation does not match.'),
            'new_password_confirmation.required' =>  $keywords['The password confirmation field is required.'] ?? __('The password confirmation field is required.')
        ];
        $validator = Validator::make($request->all(), $rules, $messages);
        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator);
        }
        $user = Auth::guard('customer')->user();
        $user->update([
            'password' => Hash::make($request->new_password)
        ]);
        Session::flash('success', $keywords['Password updated successfully.'] ?? __('Password updated successfully.'));
        return redirect()->back();
    }

    public function logoutSubmit(Request $request, $domain)
    {
        Auth::guard('customer')->logout();
        return redirect()->route('customer.login', getParam());
    }



    public function redirectToGoogle()
    {
        return Socialite::driver('google')->redirect();
    }

    public function handleGoogleCallback()
    {
        return $this->authUserViaProvider('google');
    }

    public function authUserViaProvider($provider)
    {
        $rootUser = getUser();
        if (Session::has('link')) {
            $redirectUrl = Session::get('link');
            Session::forget('link');
        } else {
            $redirectUrl = route('customer.dashboard', getParam());
        }
        $user = Socialite::driver($provider)->user();
        if ($provider == 'facebook') {
            $user = json_decode(json_encode($user), true);
        } elseif ($provider == 'google') {
            $user = json_decode(json_encode($user), true)['user'];
        }
        if ($provider == 'facebook') {
            $fname = $user['name'];
            $photo = $user['avatar'];
        } elseif ($provider == 'google') {
            $fname = $user['given_name'];
            $photo = $user['picture'];
        }
        $email = $user['email'];
        $provider_id = $user['id'];


        // retrieve user via the email
        $user = Customer::query()
            ->where('email', $email)
            ->where('user_id', $rootUser->id)
            ->first();

        // if it doesn't exist, store the new user's info (email, name, avatar, provider_name, provider_id)
        $time = Carbon::now();
        if (empty($user)) {
            $user = new Customer;
            $user->email = $email;
            $user->username = $fname;
            $user->image = $photo;
            $user->status = 1;
            $user->email_verified_at = $time;
            $user->user_id = $rootUser->id;
            $user->save();
        }


        // authenticate the user
        Auth::guard('customer')->login($user);
        // if user is banned
        if ($user->status == 0) {
            Auth::guard('customer')->logout();
            return redirect(route('customer.login', getParam()))->with('err', $keywords['Your account has been banned'] ?? __('Your account has been banned'));
        }
        // if logged in successfully
        return redirect($redirectUrl);
    }
}
