<?php

namespace App\Http\Controllers\Admin;

use App\Constants\Constant;
use App\Enums\ThemeVersion;
use App\Http\Controllers\Controller;
use App\Http\Controllers\Front\CheckoutController;
use App\Http\Helpers\MegaMailer;
use App\Http\Helpers\Uploader;
use App\Http\Helpers\UserPermissionHelper;
use App\Models\BasicExtended;
use App\Models\Customer;
use App\Models\Membership;
use App\Models\OfflineGateway;
use App\Models\Package;
use App\Models\PaymentGateway;
use App\Models\User\PaymentGateway as OnlineGateway;
use App\Models\User;
use App\Models\User\BasicSetting;
use App\Models\User\CustomPage\Page;
use App\Models\User\CustomPage\PageContent;
use App\Models\User\Language;
use App\Models\Language as AdminLang;
use App\Models\User\HomePage\SectionHeading;
use App\Models\User\PageHeading;
use App\Models\User\UserPermission;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;

class RegisterUserController extends Controller
{
    public $carbon_max_Date;
    public function __construct()
    {
        $this->carbon_max_Date = Carbon::create(9999, 12, 31, 23, 59, 59);
    }

    public function index(Request $request)
    {
        $term = $request->term;
        $users = User::when($term, function ($query, $term) {
            $query->where('username', 'like', '%' . $term . '%')->orWhere('email', 'like', '%' . $term . '%');
        })->orderBy('id', 'DESC')->paginate(10);

        $online = PaymentGateway::query()->where('status', 1)->get();
        $offline = OfflineGateway::where('status', 1)->get();
        $gateways = $online->merge($offline);
        $packages = Package::query()->where('status', '1')->get();

        return view('admin.register_user.index', compact('users', 'gateways', 'packages'));
    }

    public function view($id)
    {

        $user = User::findOrFail($id);
        $packages = Package::query()->where('status', '1')->get();
        $online = PaymentGateway::query()->where('status', 1)->get();
        $offline = OfflineGateway::where('status', 1)->get();
        $gateways = $online->merge($offline);
        return view('admin.register_user.details', compact('user', 'packages', 'gateways'));
    }

    public function editUser($id)
    {

        $user = User::findOrFail($id);
        return view('admin.register_user.edit_user', compact('user'));
    }

    public function updateUser(Request $request)
    {

        $user = User::where('id', $request->user_id)->first();;
        $rules = [
            'email' => ['email', Rule::unique('users', 'email')->ignore($user->id)]
        ];
        $messages = [
            'email.unique' => __('The email has already taken'),
        ];
        $validator = Validator::make($request->all(), $rules, $messages);
        if ($validator->fails()) {
            $validator->getMessageBag()->add('error', 'true');
            return response()->json($validator->errors());
        }

        $user->first_name = $request->first_name;
        $user->last_name = $request->last_name;
        $user->email =  $request->email;
        $user->company_name = $request->company_name;
        $user->phone = $request->phone;
        $user->city = $request->city;
        $user->state = $request->state;
        $user->address = $request->address;
        $user->country = $request->country;
        $user->update();

        Session::flash('success', __('Updated Successfully'));
        return "success";
    }

    public function store(Request $request)
    {


        if (session()->has('lang')) {
            $currentLang = AdminLang::where('code', session()->get('lang'))->first();
        } else {
            $currentLang = AdminLang::where('is_default', 1)->first();
        }
        $bs = $currentLang->basic_setting;
        $be = $currentLang->basic_extended;

        $rules = [
            'username' => 'required|alpha_num|unique:users',
            'email' => 'required|email|unique:users',
            'password' => 'required|confirmed',
            'package_id' => 'required',
            'payment_gateway' => 'required',
            'online_status' => 'required',
            'password_confirmation' => 'required'
        ];
        $messages = [
            'package_id.required' => __('The package field is required'),
            'online_status.required' => __('The publicly hidden field is required')
        ];
        $validator = Validator::make($request->all(), $rules, $messages);
        if ($validator->fails()) {
            $validator->getMessageBag()->add('error', 'true');
            return response()->json($validator->errors());
        }

        $user = User::where('username', $request['username']);
        if ($user->count() == 0) {
            $user = User::create([
                'email' => $request['email'],
                'username' => $request['username'],
                'password' => bcrypt($request['password']),
                'online_status' => $request["online_status"],
                'status' => 1,
                'email_verified' => 1,
                'company_name' => ucfirst($request['username'])
            ]);

            BasicSetting::create([
                'user_id' => $user->id,
                'theme_version' => ThemeVersion::ThemeOne->value,
                'email' => $user->email,
                'smtp_status' => 1,
                'email_address' => $user->email,
                'website_title' => $bs->website_title,
                'address' =>  null,
                'base_currency_symbol_position' => $be->base_currency_symbol_position,
                'base_currency_symbol' => $be->base_currency_symbol ?? $be->base_currency_text,
                'base_currency_text' => $be->base_currency_text,
                'base_currency_text_position' => $be->base_currency_text_position,
                'base_currency_rate' => $be->base_currency_rate,
                'primary_color' => 'FEA116',
                'secondary_color' => '0F172B',
                'breadcrumb_overlay_color' => '0F172B',
                'breadcrumb_overlay_opacity' => 0.76,
                'contact_number' => null,
                'user_id' => $user->id,
                'reply_to' => $request['email'],
                'from_name' => 'Hotelia',
                'user_id' => $user->id,
                'room_guest_checkout_status' => 1,
                'package_guest_checkout_status' => 1

            ]);

            // create payment gateways
            $payment_keywords = ['flutterwave', 'razorpay', 'paytm', 'paystack', 'instamojo', 'stripe', 'paypal', 'mollie', 'mercadopago', 'authorize.net'];
            foreach ($payment_keywords as $key => $value) {
                OnlineGateway::create([
                    'title' => null,
                    'user_id' => $user->id,
                    'details' => null,
                    'keyword' => $value,
                    'subtitle' => null,
                    'name' => ucfirst($value),
                    'type' => 'automatic',
                    'information' => null,
                    'status' => 0
                ]);
            }
        }
        if ($user) {
            $checkoutController = new CheckoutController;
            $checkoutController->insertMailTemplate($user);
            $deLang = Language::firstOrFail();
            // language wise dulicate data
            $this->tenant_lanuage_related_table_create($user);
            $package = Package::find($request['package_id']);
            $be = BasicExtended::first();
            $bs = BasicSetting::select('website_title')->first();
            $transaction_id = UserPermissionHelper::uniqidReal(8);
            $startDate = Carbon::today()->format('Y-m-d');
            if ($package->term === "monthly") {
                $endDate = Carbon::today()->addMonth()->format('Y-m-d');
            } elseif ($package->term === "yearly") {
                $endDate = Carbon::today()->addYear()->format('Y-m-d');
            } elseif ($package->term === "lifetime") {
                $endDate = Carbon::now()->addYears(999)->format('d-m-Y');
            }
            $memb = Membership::create([
                'package_price' => $package->price,
                'price' => $package->price,
                'currency' => $be->base_currency_text ? $be->base_currency_text : "USD",
                'currency_symbol' => $be->base_currency_symbol ? $be->base_currency_symbol : $be->base_currency_text,
                'payment_method' => $request["payment_gateway"],
                'transaction_id' => $transaction_id ? $transaction_id : 0,
                'status' => 1,
                'is_trial' => 0,
                'trial_days' => 0,
                'receipt' => $request["receipt_name"] ? $request["receipt_name"] : null,
                'transaction_details' => null,
                'settings' => json_encode($be),
                'package_id' => $request['package_id'],
                'user_id' => $user->id,
                'start_date' => Carbon::parse($startDate),
                'expire_date' => Carbon::parse($endDate),
            ]);
            $package = Package::findOrFail($request['package_id']);
            $features = json_decode($package->features, true);
            $features[] = "Contact";
            UserPermission::create([
                'package_id' => $request['package_id'],
                'user_id' => $user->id,
                'permissions' => json_encode($features)
            ]);

            User\Section::create([
                'user_id' => $user->id
            ]);

            $requestData = [
                'start_date' => $startDate,
                'expire_date' => $endDate,
                'payment_method' => $request['payment_gateway']
            ];
            $file_name = $this->makeInvoice($requestData, "membership", $user, null, $package->price, $request['payment_gateway'], null, $be->base_currency_symbol_position, $be->base_currency_symbol, $be->base_currency_text, $transaction_id, $package->title, $memb);

            $mailer = new MegaMailer();
            $startDate = Carbon::parse($startDate);
            $endDate = Carbon::parse($endDate);
            $data = [
                'toMail' => $user->email,
                'toName' => $user->fname,
                'username' => $user->username,
                'package_title' => $package->title,
                'package_price' => ($be->base_currency_text_position == 'left' ? $be->base_currency_text . ' ' : '') . $package->price . ($be->base_currency_text_position == 'right' ? ' ' . $be->base_currency_text : ''),
                'activation_date' => $startDate->toFormattedDateString(),
                'expire_date' => $endDate->toFormattedDateString(),
                'membership_invoice' => $file_name,
                'website_title' => $bs->website_title,
                'templateType' => 'registration_with_premium_package',
                'type' => 'registrationWithPremiumPackage'
            ];
            $mailer->mailFromAdmin($data);
        }
        Session::flash('success', __('Created Successfully'));
        return "success";
    }

    public function userban(Request $request)
    {
        $user = User::where('id', $request->user_id)->first();
        $user->update([
            'status' => $request->status,
        ]);
        Session::flash('success', __('Updated Successfully'));
        return back();
    }

    public function userPublicly(Request $request)
    {
        $user = User::where('id', $request->user_id)->first();
        $user->update([
            'online_status' => $request->online_status,
        ]);
        Session::flash('success', __('Updated Successfully'));
        return back();
    }

    public function emailStatus(Request $request)
    {
        $user = User::findOrFail($request->user_id);
        $user->update([
            'email_verified' => $request->email_verified,
        ]);
        Session::flash('success', __('Email status updated for') . ' ' . $user->username);
        return back();
    }

    public function userFeatured(Request $request)
    {
        $user = User::where('id', $request->user_id)->first();
        $user->featured = $request->featured;
        $user->save();
        Session::flash('success', __('Updated Successfully'));
        return back();
    }

    public function changePass($id)
    {
        $data['user'] = User::findOrFail($id);
        return view('admin.register_user.password', $data);
    }

    public function updatePassword(Request $request)
    {
        $request->validate([
            'new_password' => 'required',
            'confirm_password' => 'required',
        ]);

        $user = User::findOrFail($request->user_id);
        if ($request->new_password == $request->confirm_password) {
            $input['password'] = Hash::make($request->new_password);
        } else {
            return back()->with('warning', __('Confirm password does not match'));
        }
        $user->update($input);
        Session::flash('success', __('Updated Successfully'));
        return 'success';
    }

    public function delete(Request $request)
    {
        $this->delete_user($request->user_id);
        Session::flash('success', __('Deleted Successfully'));
        return back();
    }

    public function bulkDelete(Request $request)
    {
        $ids = $request->ids;
        foreach ($ids as $id) {
            $this->delete_user($id);
        }
        Session::flash('success', __('Deleted Successfully'));
        return "success";
    }

    public function delete_user($id)
    {
        $user = User::query()->findOrFail($id);
        $bss = BasicSetting::query()->where('user_id', $user->id)->first();

        /**
         * delete 'page heading' info
         */
        $page_headings = $user->page_headings()->get();
        if ($page_headings->count() > 0) {
            foreach ($page_headings as $heading) {
                $heading->delete();
            }
        }


        /**
         * delete room,roomContent,booking
         */
        $userRooms = $user->rooms()->get();
        if ($userRooms->count() > 0) {
            foreach ($userRooms as  $userRoom) {
                if ($userRoom->roomContent->count() > 0) {
                    foreach ($userRoom->roomContent as $room_content) {
                        $room_content->delete();
                    }
                }
                Uploader::remove(public_path(Constant::WEBSITE_ROOM_IMAGE), $userRoom->featured_img);
                foreach (json_decode($userRoom->slider_imgs, true) as $slider) {
                    Uploader::remove(public_path(Constant::WEBSITE_ROOM_SLIDER_IMAGE), $slider);
                }
                $userRoom->delete();
            }
        }

        /**
         * delete room booking
         */

        if ($user->bookHotelRoom->count() > 0) {
            foreach ($user->bookHotelRoom as $room_booking) {
                Uploader::remove(public_path(Constant::WEBSITE_ROOM_BOOKING_ATTACHMENTS), $room_booking->attachment);
                Uploader::remove(public_path(Constant::WEBSITE_ROOM_BOOKING_INVOICE), $room_booking->invoice);
                $room_booking->delete();
            }
        }

        /**
         * delete  RoomAmenity
         */

        $roomAmenities = $user->room_amenities()->get();
        if ($roomAmenities->count() > 0) {
            foreach ($roomAmenities as $amenity) {
                $amenity->delete();
            }
        }


        /**
         * delete  RoomCoupon
         */
        $roomCoupons = $user->roomBookingCoupons()->get();
        if ($roomCoupons->count() > 0) {
            foreach ($roomCoupons as $coupon) {
                $coupon->delete();
            }
        }

        /**
         * delete room category
         */
        $roomCategoris = $user->room_categories()->get();
        if ($roomCategoris->count() > 0) {
            foreach ($roomCategoris as $cat) {
                $cat->delete();
            }
        }

        /**
         * delete room review
         */

        $roomReviews = $user->room_reviews()->get();
        if ($roomReviews->count() > 0) {
            foreach ($roomReviews as $roomReview) {
                $roomReview->delete();
            }
        }

        /**
         * delete package,packageContent,booking
         */

        $userPackages = $user->tourPackages()->get();

        if ($userPackages->count() > 0) {
            foreach ($userPackages as  $userPackage) {
                if ($userPackage->packageContent->count() > 0) {
                    foreach ($userPackage->packageContent as $packageContent) {
                        $packageContent->delete();
                    }
                }
                Uploader::remove(public_path(Constant::WEBSITE_PACKAGE_IMAGE), $userPackage->featured_img);
                foreach (json_decode($userPackage->slider_imgs, true) as $slider) {
                    Uploader::remove(public_path(Constant::WEBSITE_PACKAGE_SLIDER_IMAGE), $slider);
                }
                $userPackage->delete();
            }
        }

        /**
         * delete packag booking
         */

        if ($user->tourPackageBookings->count() > 0) {
            foreach ($user->tourPackageBookings as $tourPackageBooking) {
                Uploader::remove(public_path(Constant::WEBSITE_PACKAGE_BOOKING_ATTACHMENTS), $tourPackageBooking->attachment);
                Uploader::remove(public_path(Constant::WEBSITE_PACKAGE_BOOKING_INVOICE), $tourPackageBooking->invoice);
                $tourPackageBooking->delete();
            }
        }

        /**
         * delete packag category
         */

        if ($user->packageCategories->count() > 0) {
            foreach ($user->packageCategories as $packCate) {
                $packCate->delete();
            }
        }

        /**
         * delete packag coupon
         */
        if ($user->packageCoupons->count() > 0) {
            foreach ($user->packageCoupons as $packCoupon) {
                $packCoupon->delete();
            }
        }


        /**
         * delete tour package review
         */

        if ($user->giveReviewForPackage()->count() > 0) {
            foreach ($user->giveReviewForPackage as $packageReview) {
                $packageReview->delete();
            }
        }

        /**
         * delete tour package locations
         */

        if ($user->packageLocations()->count() > 0) {
            foreach ($user->packageLocations as $location) {
                $location->delete();
            }
        }


        /**
         * delete tour package plans
         */

        if ($user->packagePlans()->count() > 0) {
            foreach ($user->packagePlans as $plan) {
                $plan->delete();
            }
        }




        // /**
        //  * delete 'custom domains' info
        //  */

        $custom_domains = $user->custom_domains()->get();
        if ($custom_domains->count() > 0) {
            foreach ($custom_domains as $custom_domain) {
                $custom_domain->delete();
            }
        }

        /**
         * delete 'memberships' info
         */


        $memberships = $user->memberships()->get();
        if ($memberships->count() > 0) {
            foreach ($memberships as $membership) {

                if (!is_null($membership->receipt)) {
                    @unlink(public_path('assets/front/img/membership/receipt/' . $membership->receipt));
                }
                $membership->delete();
            }
        }

        /**
         * delete 'QR Codes' info
         */

        if ($user->qr_codes()->count() > 0) {

            $qr_codes = $user->qr_codes()->get();
            foreach ($qr_codes as $qr) {
                Uploader::remove(public_path(Constant::WEBSITE_QRCODE_IMAGE), $qr->image);
                $qr->delete();
            }
        }

        if ($user->sections->count() > 0) {
            $user->sections()->delete();
        }

        /**
         * delete 'count information's' info
         */

        $count_informations = $user->counterInformations()->get();
        if ($count_informations->count() > 0) {
            foreach ($count_informations as $count_information) {
                $count_information->delete();
            }
        }

        /**
         * delete 'user faqs' info
         */

        $faqs = $user->faqs()->get();
        if ($faqs->count() > 0) {
            foreach ($faqs as $faq) {
                $faq->delete();
            }
        }

        /**
         * delete 'user seos' info
         */

        $seos = $user->seos()->get();
        if ($seos->count() > 0) {
            foreach ($seos as $seo) {
                $seo->delete();
            }
        }

        /**
         * delete 'blog infos'
         */

        $userBlogs = $user->blogs()->get();

        if (count($userBlogs) > 0) {
            foreach ($userBlogs as $blogData) {
                if (!is_null($blogData->image)) {
                    Uploader::remove(public_path(Constant::WEBSITE_BLOG_IMAGE),  $blogData->image);
                }
                $blogData->delete();
            }
        }
        
        $userBlogInfo = $user->blog_contents()->get();
        if (count($userBlogInfo) > 0) {
            foreach ($userBlogInfo as $info) {
                $info->delete();
            }
        }


        /**
         * delete 'blog categories' info
         */

        if ($user->blog_categories()->count() > 0) {
            $user->blog_categories()->delete();
        }

        /**
         * delete 'social medias' info
         */

        if ($user->social_media()->count() > 0) {
            $user->social_media()->delete();
        }

        /**
         * delete 'languages' info
         */

        if ($user->languages()->count() > 0) {
            $user->languages()->delete();
        }

        /**
         * delete 'footer quick links' info
         */

        if ($user->footer_quick_links()->count() > 0) {
            $user->footer_quick_links()->delete();
        }

        // /**
        //  * delete 'subscribers' info
        //  */

        if ($user->subscribers()->count() > 0) {
            $user->subscribers()->delete();
        }

        // /**
        //  * delete 'footer texts' info
        //  */

        if ($user->footer_texts()->count() > 0) {
            $user->footer_texts()->delete();
        }


        // /**
        //  * delete 'facilities' info
        //  */

        if ($user->facilities()->count() > 0) {
            $user->facilities()->delete();
        }

        /**
         * delete gallery
         */

        $galleries = $user->galleries()->get();
        foreach ($galleries as $galry) {
            if (!is_null($galry->gallery_img)) {
                Uploader::remove(public_path(Constant::WEBSITE_GALLERY_IMAGE), $galry->gallery_img);
            }
            $galry->delete();
        }


        /**
         * delete HeroSlider
         */

        $heroSliders = $user->hero_sliders()->get();
        foreach ($heroSliders as $slider) {
            if (!is_null($slider->img)) {
                Uploader::remove(public_path(Constant::WEBSITE_HERO_SLIDER_IMAGE), $slider->img);
            }
            $slider->delete();
        }


        /**
         * delete Brand
         */

        $brands = $user->brands()->get();
        foreach ($brands as $brand) {
            if (!is_null($brand->brand_img)) {
                Uploader::remove(public_path(Constant::WEBSITE_BRAND_IMAGE), $brand->brand_img);
            }
            $brand->delete();
        }


        /**
         * delete gallery category
         */

        $galCats = $user->gallery_categories()->get();
        foreach ($galCats as $galcat) {
            $galcat->delete();
        }


        /**
         * delete tickets
         */

        $userTickets = $user->tickets()->with('messages')->get();
        if ($userTickets->count() > 0) {
            foreach ($userTickets as $userTicket) {
                foreach ($userTicket->messages as $mess) {
                    $mess->delete();
                }
                $userTicket->delete();
            }
        }

        /**
         * delete IntroCountInfos
         */

        $introCountInfos = $user->counterInfos()->get();
        if ($introCountInfos->count() > 0) {
            foreach ($introCountInfos as $intro) {
                $intro->delete();
            }
        }

        /**
         * delete UserPermissions
         */

        $userPermissions = $user->permissions()->get();
        if ($userPermissions->count() > 0) {
            foreach ($userPermissions as $permission) {
                $permission->delete();
            }
        }


        /**
         * delete user service,serviceContent
         */

        $userServices = $user->services()->get();
        if ($userServices->count() > 0) {
            foreach ($userServices as $service) {
                if ($service->serviceContent->count() > 0) {
                    foreach ($service->serviceContent as $serContent) {
                        $serContent->delete();
                    }
                }
                $service->delete();
            }
        }

        /**
         * delete 'user testimonials' info
         */

        if ($user->testimonials()->count() > 0) {

            $testimonials = $user->testimonials()->get();
            foreach ($testimonials as $testimonial) {
                if (!is_null($testimonial->image)) {
                    Uploader::remove(public_path(Constant::WEBSITE_TESTIMONIAL_SECTION_IMAGE), $testimonial->image);
                }
                $testimonial->delete();
            }
        }
        

        $sections = $user->section_headings()->get();
        foreach ($sections as $section) {
            if (!is_null($section)) {
                if (!is_null($section->faq_section_image)) {
                    Uploader::remove(public_path(Constant::WEBSITE_FAQ_SECTION_IMAGE), $section->faq_section_image);
                }

                if (!is_null($section->testimonial_section_image)) {
                    Uploader::remove(public_path(Constant::WEBSITE_TESTIMONIAL_SECTION_IMAGE), $section->testimonial_section_image);
                }

                if (!is_null($section->video_section_image)) {
                    Uploader::remove(public_path(Constant::WEBSITE_VIDEO_SECTION_IMAGE), $section->video_section_image);
                }

                if (!is_null($section->facility_section_image)) {
                    Uploader::remove(public_path(Constant::WEBSITE_FACILITY_SECTION_IMAGE), $section->facility_section_image);
                }

                if (!is_null($section->faq_section_image)) {
                    Uploader::remove(public_path(Constant::WEBSITE_FAQ_SECTION_IMAGE), $section->faq_section_image);
                }

                if (!is_null($section->intro_img)) {
                    Uploader::remove(public_path(Constant::WEBSITE_INTRO_SECTION_IMAGE), $section->intro_img);
                }

                if (!is_null($section->hero_section_image)) {
                    Uploader::remove(public_path(Constant::WEBSITE_HERO_SECTION_IMAGE), $section->hero_section_image);
                }

                if (!is_null($section->about_page_about_section_image)) {
                    Uploader::remove(public_path(Constant::WEBSITE_ABOUT_US_SECTION_IMAGE), $section->about_page_about_section_image);
                }
                if (!is_null($section->about_page_facility_section_image)) {
                    Uploader::remove(public_path(Constant::WEBSITE_FACILITY_SECTION_IMAGE), $section->about_page_facility_section_image);
                }
            }
            $section->delete();
        }



        /**
         * delete 'menus' info
         */



        if ($user->menus()->count() > 0) {
            foreach ($user->menus as $menu) {
                $menu->delete();
            }
        }


        /**
         * delete 'popup' infos
         */
        $popups = $user->announcementPopup()->get();
        if (count($popups) > 0) {
            foreach ($popups as $popup) {
                if (!is_null($popup->image)) {
                    Uploader::remove(public_path(Constant::WEBSITE_ANNOUNCEMENT_POPUP_IMAGE), $popup->image);
                }
                $popup->delete();
            }
        }

        /**
         * delete 'page contents'
         */

        $customPageInfos = $user->customPageInfo()->get();
        if (count($customPageInfos) > 0) {
            foreach ($customPageInfos as $customPageData) {
                $customPageInfo = $customPageData;
                $customPageData->delete();

                // delete the 'custom page' if, this page does not contain any other page content in any other language
                $otherPageContents = PageContent::query()
                    ->where('user_id', $user->id)
                    ->where('page_id', '=', $customPageInfo->page_id)
                    ->get();

                if (count($otherPageContents) == 0) {
                    $page = Page::query()
                        ->where('user_id', $user->id)
                        ->find($customPageInfo->page_id);
                    $page->delete();
                }
            }
        }

        /**
         * delete 'newsletter section' info
         */

        $newsletterSecs = $user->newsletterSection()->get();
        foreach ($newsletterSecs as $newsletterSec) {
            if (!empty($newsletterSec)) {
                Uploader::remove(public_path(Constant::WEBSITE_NEWSLETTER_SECTION_IMAGE), $newsletterSec->background_image, $bss, $user->id);
                Uploader::remove(public_path(Constant::WEBSITE_NEWSLETTER_SECTION_IMAGE), $newsletterSec->image, $bss, $user->id);
                $newsletterSec->delete();
            }
        }

        /**
         * delete 'about us section' info
         */
        $about_us_sections = $user->about_us_section()->get();
        foreach ($about_us_sections as $about_us_section) {
            if (!empty($about_us_section)) {
                Uploader::remove(public_path(Constant::WEBSITE_ABOUT_US_SECTION_IMAGE), $about_us_section->image, $bss, $user->id);
                $about_us_section->delete();
            }
        }

        /**
         * delete 'features section' info
         */

        $features = $user->features()->get();
        foreach ($features as $feature) {
            if (!empty($feature)) {
                Uploader::remove(public_path(Constant::WEBSITE_FEATURE_SECTION_IMAGE), $feature->features_section_image, $bss, $user->id);
                $feature->delete();
            }
        }

        /**
         * delete 'online gateways' info
         */

        $online_gateways = $user->online_gateways()->get();
        foreach ($online_gateways as $online_gateway) {
            if (!empty($online_gateway)) {
                $online_gateway->delete();
            }
        }

        /**
         * delete 'offline gateways' info
         */

        $offline_gateways = $user->offline_gateways()->get();
        foreach ($offline_gateways as $offline_gateway) {
            if (!empty($offline_gateway)) {
                $offline_gateway->delete();
            }
        }

        /**
         * delete 'mail templates' info
         */

        $mail_templates = $user->mail_templates()->get();
        if (!empty($mail_templates)) {
            foreach ($mail_templates as $mt) {
                if (!empty($mt)) {
                    $mt->delete();
                }
            }
        }

        /**
         * delete 'customer details' info
         */

        $customers = Customer::where('user_id', $user->id)->get();
        foreach ($customers as $customer) {
            // delete customer image
            Uploader::remove(public_path(Constant::WEBSITE_TENANT_CUSTOMER_IMAGE), $customer->image);
            $customer->delete();
        }

        /**
         * delete 'advertisements' info
         */

        $ads = $user->advertisements()->get();
        foreach ($ads as $ad) {
            // delete customer image
            Uploader::remove(public_path(Constant::WEBSITE_ADVERTISEMENT_IMAGE), $ad->image);
            $ad->delete();
        }
        
        /**
         * delete 'additional sections' info
         */
        $additional_sections = $user->additional_sections()->get();
        foreach ($additional_sections as $section) {
            $section->page_content()->delete();
            $section->delete();
        }


        /**
         * delete 'basic extends' info
         */

        $basic_extendeds = $user->basic_extendeds()->get();
        foreach ($basic_extendeds as $be) {
            if (!is_null($be)) {
                Uploader::remove(public_path(Constant::WEBSITE_ERROR_IMAGE), $be->page_not_found_image_one, $bss, $user->id);
                Uploader::remove(public_path(Constant::WEBSITE_ERROR_IMAGE), $be->page_not_found_image_two, $bss, $user->id);
                $be->delete();
            }
        }


        /**
         * delete 'basic settings' info
         */

        $bs = $user->basic_setting()->first();
        if (!empty($bs)) {
            Uploader::remove(public_path(Constant::WEBSITE_FAVICON), $bs->favicon, $bss, $user->id);
            Uploader::remove(public_path(Constant::WEBSITE_LOGO), $bs->logo, $bss, $user->id);
            Uploader::remove(public_path(Constant::WEBSITE_MAINTENANCE_IMAGE), $bs->maintenance_img, $bss, $user->id);
            Uploader::remove(public_path(Constant::WEBSITE_BREADCRUMB), $bs->breadcrumb, $bss, $user->id);
            Uploader::remove(public_path(Constant::WEBSITE_FOOTER_LOGO), $bs->footer_logo, $bss, $user->id);
            Uploader::remove(public_path(Constant::WEBSITE_QRCODE_IMAGE), $bs->qr_image, $bss, $user->id);
            Uploader::remove(public_path(Constant::WEBSITE_QRCODE_IMAGE), $bs->qr_inserted_image, $bss, $user->id);
            $bs->delete();
        }

        //user profile image
        if (!is_null($user->photo)) {
            Uploader::remove(public_path(Constant::WEBSITE_TENANT_IMAGE), $user->photo);
            @unlink(public_path('assets/tenant/img/users/' . $user->photo));
        }
        $user->delete();
        return;
    }

    public function removeCurrPackage(Request $request)
    {

        $userId = $request->user_id;
        $user = User::findOrFail($userId);
        $currMembership = UserPermissionHelper::currMembOrPending($userId);
        $currPackage = Package::select('title')->findOrFail($currMembership->package_id);
        $nextMembership = UserPermissionHelper::nextMembership($userId);
        $be = BasicExtended::first();
        $bs = BasicSetting::select('website_title')->first();
        $today = Carbon::now();
        // just expire the current package
        $currMembership->expire_date = $today->subDay();
        $currMembership->modified = 1;
        if ($currMembership->status == 0) {
            $currMembership->status = 2;
        }
        $currMembership->save();

        // if next package exists
        if (!empty($nextMembership)) {
            $nextPackage = Package::find($nextMembership->package_id);
            $nextMembership->start_date = Carbon::parse(Carbon::today()->format('d-m-Y'));
            if ($nextPackage->term == 'monthly') {
                $nextMembership->expire_date = Carbon::parse(Carbon::today()->addMonth()->format('d-m-Y'));
            } elseif ($nextPackage->term == 'yearly') {
                $nextMembership->expire_date = Carbon::parse(Carbon::today()->addYear()->format('d-m-Y'));
            } elseif ($nextPackage->term == 'lifetime') {
                $nextMembership->expire_date = Carbon::parse($this->carbon_max_Date->format('d-m-Y'));
            }
            $nextMembership->save();
            $features = json_decode($nextPackage->features, true);
            $features[] = "Contact";
            UserPermission::where('user_id', $user->id)->update([
                'package_id' => $nextPackage->id,
                'user_id' => $user->id,
                'permissions' => json_encode($features)
            ]);
        }
        $this->sendMail(NULL, NULL, $request->payment_method, $user, $bs, $be, 'admin_removed_current_package', NULL, $currPackage->title);
        Session::flash('success', __('Deleted Successfully'));
        return back();
    }


    public function sendMail($memb, $package, $paymentMethod, $user, $bs, $be, $mailType, $replacedPackage = NULL, $removedPackage = NULL)
    {
        if ($mailType != 'admin_removed_current_package' && $mailType != 'admin_removed_next_package') {
            $transaction_id = UserPermissionHelper::uniqidReal(8);
            $activation = $memb->start_date;
            $expire = $memb->expire_date;
            $info['start_date'] = $activation->toFormattedDateString();
            $info['expire_date'] = $expire->toFormattedDateString();
            $info['payment_method'] = $paymentMethod;
            $file_name = $this->makeInvoice($info, "membership", $user, NULL, $package->price, "Stripe", $user->phone, $be->base_currency_symbol_position, $be->base_currency_symbol, $be->base_currency_text, $transaction_id, $package->title, $memb);
        }

        $mailer = new MegaMailer();
        $data = [
            'toMail' => $user->email,
            'toName' => $user->fname,
            'username' => $user->username,
            'website_title' => $bs->website_title,
            'templateType' => $mailType
        ];

        if ($mailType != 'admin_removed_current_package' && $mailType != 'admin_removed_next_package') {
            $data['package_title'] = $package->title;
            $data['package_price'] = ($be->base_currency_text_position == 'left' ? $be->base_currency_text . ' ' : '') . $package->price . ($be->base_currency_text_position == 'right' ? ' ' . $be->base_currency_text : '');
            $data['activation_date'] = $activation->toFormattedDateString();
            $data['expire_date'] = Carbon::parse($expire->toFormattedDateString())->format('Y') == '9999' ? 'Lifetime' : $expire->toFormattedDateString();
            $data['membership_invoice'] = $file_name;
        }
        if ($mailType != 'admin_removed_current_package' || $mailType != 'admin_removed_next_package') {
            $data['removed_package_title'] = $removedPackage;
        }

        if (!empty($replacedPackage)) {
            $data['replaced_package'] = $replacedPackage;
        }
        $mailer->mailFromAdmin($data);
    }

    public function changeCurrPackage(Request $request)
    {
        $userId = $request->user_id;
        $user = User::findOrFail($userId);
        $currMembership = UserPermissionHelper::currMembOrPending($userId);
        $nextMembership = UserPermissionHelper::nextMembership($userId);
        $be = BasicExtended::first();
        $bs = BasicSetting::select('website_title')->first();
        $selectedPackage = Package::find($request->package_id);
        // if the user has a next package to activate & selected package is 'lifetime' package
        if (!empty($nextMembership) && $selectedPackage->term == 'lifetime') {
            Session::flash('membership_warning', __('To add a Lifetime package as Current Package, You have to remove the next package'));
            return back();
        }
        // expire the current package
        $currMembership->expire_date = Carbon::parse(Carbon::now()->subDay()->format('d-m-Y'));
        $currMembership->modified = 1;
        if ($currMembership->status == 0) {
            $currMembership->status = 2;
        }
        $currMembership->save();

        // calculate expire date for selected package
        if ($selectedPackage->term == 'monthly') {
            $exDate = Carbon::now()->addMonth()->format('d-m-Y');
        } elseif ($selectedPackage->term == 'yearly') {
            $exDate = Carbon::now()->addYear()->format('d-m-Y');
        } elseif ($selectedPackage->term == 'lifetime') {
            $exDate = Carbon::now()->addYears(999)->format('d-m-Y');
        }
        // store a new membership for selected package
        $selectedMemb = Membership::create([
            'price' => $selectedPackage->price,
            'currency' => $be->base_currency_text,
            'currency_symbol' => $be->base_currency_symbol,
            'payment_method' => $request->payment_method,
            'transaction_id' => uniqid(),
            'status' => 1,
            'receipt' => NULL,
            'transaction_details' => NULL,
            'settings' => json_encode($be),
            'package_id' => $selectedPackage->id,
            'user_id' => $userId,
            'start_date' => Carbon::parse(Carbon::now()->format('d-m-Y')),
            'expire_date' => Carbon::parse($exDate),
            'is_trial' => 0,
            'trial_days' => 0,
        ]);

        $features = json_decode($selectedPackage->features, true);
        $features[] = "Contact";
        UserPermission::where('user_id', $user->id)->update([
            'package_id' => $request['package_id'],
            'user_id' => $user->id,
            'permissions' => json_encode($features)
        ]);

        // if the user has a next package to activate & selected package is not 'lifetime' package
        if (!empty($nextMembership) && $selectedPackage->term != 'lifetime') {
            $nextPackage = Package::find($nextMembership->package_id);

            // calculate & store next membership's start_date
            $nextMembership->start_date = Carbon::parse(Carbon::parse($exDate)->addDay()->format('d-m-Y'));
            // calculate & store expire date for next membership
            if ($nextPackage->term == 'monthly') {
                $exDate = Carbon::parse(Carbon::parse(Carbon::parse($exDate)->addDay()->format('d-m-Y'))->addMonth()->format('d-m-Y'));
            } elseif ($nextPackage->term == 'yearly') {
                $exDate = Carbon::parse(Carbon::parse(Carbon::parse($exDate)->addDay()->format('d-m-Y'))->addYear()->format('d-m-Y'));
            } else {
                $exDate = Carbon::parse($this->carbon_max_Date->format('d-m-Y'));
            }
            $nextMembership->expire_date = $exDate;
            $nextMembership->save();
        }
        $currentPackage = Package::select('title')->findOrFail($currMembership->package_id);
        $this->sendMail($selectedMemb, $selectedPackage, $request->payment_method, $user, $bs, $be, 'admin_changed_current_package', $currentPackage->title);

        Session::flash('success', __('Changed Successfully'));
        return back();
    }

    public function addCurrPackage(Request $request)
    {
        $userId = $request->user_id;
        $user = User::findOrFail($userId);
        $be = BasicExtended::first();
        $bs = BasicSetting::select('website_title')->first();
        $selectedPackage = Package::find($request->package_id);

        // calculate expire date for selected package
        if ($selectedPackage->term == 'monthly') {
            $exDate = Carbon::now()->addMonth()->format('d-m-Y');
        } elseif ($selectedPackage->term == 'yearly') {
            $exDate = Carbon::now()->addYear()->format('d-m-Y');
        } elseif ($selectedPackage->term == 'lifetime') {
            $exDate = Carbon::now()->addYears(999)->format('d-m-Y');
        }
        // store a new membership for selected package
        $selectedMemb = Membership::create([
            'price' => $selectedPackage->price,
            'currency' => $be->base_currency_text,
            'currency_symbol' => $be->base_currency_symbol,
            'payment_method' => $request->payment_method,
            'transaction_id' => uniqid(),
            'status' => 1,
            'receipt' => NULL,
            'transaction_details' => NULL,
            'settings' => json_encode($be),
            'package_id' => $selectedPackage->id,
            'user_id' => $userId,
            'start_date' => Carbon::parse(Carbon::now()->format('d-m-Y')),
            'expire_date' => Carbon::parse($exDate),
            'is_trial' => 0,
            'trial_days' => 0,
        ]);

        $features = json_decode($selectedPackage->features, true);
        $features[] = "Contact";
        UserPermission::where('user_id', $user->id)->update([
            'package_id' => $request['package_id'],
            'user_id' => $user->id,
            'permissions' => json_encode($features)
        ]);
        $this->sendMail($selectedMemb, $selectedPackage, $request->payment_method, $user, $bs, $be, 'admin_added_current_package');
        Session::flash('success', __('Added Successfully'));
        return back();
    }

    public function removeNextPackage(Request $request)
    {
        $userId = $request->user_id;
        $user = User::findOrFail($userId);
        $be = BasicExtended::first();
        $bs = BasicSetting::select('website_title')->first();
        $nextMembership = UserPermissionHelper::nextMembership($userId);
        // set the start_date to unlimited
        $nextMembership->start_date = Carbon::parse($this->carbon_max_Date->format('d-m-Y'));
        $nextMembership->modified = 1;
        $nextMembership->save();
        $nextPackage = Package::select('title')->findOrFail($nextMembership->package_id);
        $this->sendMail(NULL, NULL, $request->payment_method, $user, $bs, $be, 'admin_removed_next_package', NULL, $nextPackage->title);

        Session::flash('success', __('Removed Successfully'));
        return back();
    }

    public function changeNextPackage(Request $request)
    {
        $userId = $request->user_id;
        $user = User::findOrFail($userId);
        $bs = BasicSetting::select('website_title')->first();
        $be = BasicExtended::first();
        $nextMembership = UserPermissionHelper::nextMembership($userId);
        $nextPackage = Package::find($nextMembership->package_id);
        $selectedPackage = Package::find($request->package_id);

        $prevStartDate = $nextMembership->start_date;


        // set the start_date to unlimited
        $nextMembership->start_date = Carbon::parse($this->carbon_max_Date->format('d-m-Y'));
        $nextMembership->modified = 1;
        $nextMembership->save();

        // calculate expire date for selected package
        if ($selectedPackage->term == 'monthly') {
            $exDate = Carbon::parse($prevStartDate)->addMonth()->format('d-m-Y');
        } elseif ($selectedPackage->term == 'yearly') {
            $exDate = Carbon::parse($prevStartDate)->addYear()->format('d-m-Y');
        } elseif ($selectedPackage->term == 'lifetime') {
            $exDate = Carbon::parse($this->carbon_max_Date->format('d-m-Y'));
        }

        // store a new membership for selected package
        $selectedMemb = Membership::create([
            'price' => $selectedPackage->price,
            'currency' => $be->base_currency_text,
            'currency_symbol' => $be->base_currency_symbol,
            'payment_method' => $request->payment_method,
            'transaction_id' => uniqid(),
            'status' => 1,
            'receipt' => NULL,
            'transaction_details' => NULL,
            'settings' => json_encode($be),
            'package_id' => $selectedPackage->id,
            'user_id' => $userId,
            'start_date' => Carbon::parse($prevStartDate),
            'expire_date' => Carbon::parse($exDate),
            'is_trial' => 0,
            'trial_days' => 0,
        ]);

        $this->sendMail($selectedMemb, $selectedPackage, $request->payment_method, $user, $bs, $be, 'admin_changed_next_package', $nextPackage->title);

        Session::flash('success', __('Changed Successfully'));
        return back();
    }

    public function addNextPackage(Request $request)
    {
        $userId = $request->user_id;
        $hasPendingMemb = UserPermissionHelper::hasPendingMembership($userId);
        if ($hasPendingMemb) {
            Session::flash('membership_warning', __('This user already has a Pending Package. Please take an action (change / remove / approve / reject) for that package first.'));
            return back();
        }

        $currMembership = UserPermissionHelper::userPackage($userId);
        $currPackage = Package::find($currMembership->package_id);
        $be = BasicExtended::first();
        $user = User::findOrFail($userId);
        $bs = BasicSetting::select('website_title')->first();

        $selectedPackage = Package::find($request->package_id);

        if ($currMembership->is_trial == 1) {
            Session::flash('membership_warning', __('If your current package is trial package, then you have to change / remove the current package first.'));
            return back();
        }


        // if current package is not lifetime package
        if ($currPackage->term != 'lifetime') {
            // calculate expire date for selected package
            if ($selectedPackage->term == 'monthly') {
                $exDate = Carbon::parse($currMembership->expire_date)->addDay()->addMonth()->format('d-m-Y');
            } elseif ($selectedPackage->term == 'yearly') {
                $exDate = Carbon::parse($currMembership->expire_date)->addDay()->addYear()->format('d-m-Y');
            } elseif ($selectedPackage->term == 'lifetime') {
                $exDate = Carbon::parse($this->carbon_max_Date->format('d-m-Y'));
            }
            // store a new membership for selected package
            $selectedMemb = Membership::create([
                'price' => $selectedPackage->price,
                'currency' => $be->base_currency_text,
                'currency_symbol' => $be->base_currency_symbol,
                'payment_method' => $request->payment_method,
                'transaction_id' => uniqid(),
                'status' => 1,
                'receipt' => NULL,
                'transaction_details' => NULL,
                'settings' => json_encode($be),
                'package_id' => $selectedPackage->id,
                'user_id' => $userId,
                'start_date' => Carbon::parse(Carbon::parse($currMembership->expire_date)->addDay()->format('d-m-Y')),
                'expire_date' => Carbon::parse($exDate),
                'is_trial' => 0,
                'trial_days' => 0,
            ]);

            $this->sendMail($selectedMemb, $selectedPackage, $request->payment_method, $user, $bs, $be, 'admin_added_next_package');
        } else {
            Session::flash('membership_warning', __('If your current package is lifetime package, then you have to change / remove the current package first.'));
            return back();
        }

        Session::flash('success', __('Added Successfully'));
        return back();
    }

    public function userTemplate(Request $request)
    {
        if ($request->template == 1) {
            $prevImg = $request->file('preview_image');
            $allowedExts = array('jpg', 'png', 'jpeg');
            $rules = [
                'serial_number' => 'required|integer',
                'preview_image' => [
                    'required',
                    function ($attribute, $value, $fail) use ($prevImg, $allowedExts) {
                        if (!empty($prevImg)) {
                            $ext = $prevImg->getClientOriginalExtension();
                            if (!in_array($ext, $allowedExts)) {
                                return $fail(__("Only png, jpg, jpeg image is allowed"));
                            }
                        }
                    },
                ]
            ];
            $validator = Validator::make($request->all(), $rules);
            if ($validator->fails()) {
                Session::flash('warning', $validator->errors()->first());
                return redirect()->back();
            }
        }

        $user = User::where('id', $request->user_id)->first();
        if ($request->template == 1) {
            if ($request->hasFile('preview_image')) {
                @unlink(public_path('assets/front/img/template-previews/' . $user->template_img));
                $filename = uniqid() . '.' . $prevImg->getClientOriginalExtension();
                $dir = public_path('assets/front/img/template-previews/');
                @mkdir($dir, 0775, true);
                $request->file('preview_image')->move($dir, $filename);
                $user->template_img = $filename;
            }
            $user->template_serial_number = $request->serial_number;
        } else {
            @unlink(public_path('assets/front/img/template-previews/' . $user->template_img));
            $user->template_img = NULL;
            $user->template_serial_number = 0;
        }
        $user->preview_template = $request->template;
        $user->save();
        Session::flash('success', __('Updated Successfully'));
        return back();
    }

    public function userUpdateTemplate(Request $request)
    {

        $prevImg = $request->file('preview_image');
        $allowedExts = array('jpg', 'png', 'jpeg');
        $rules = [
            'serial_number' => 'required|integer',
            'preview_image' => [
                function ($attribute, $value, $fail) use ($prevImg, $allowedExts) {
                    if (!empty($prevImg)) {
                        $ext = $prevImg->getClientOriginalExtension();
                        if (!in_array($ext, $allowedExts)) {
                            return $fail(__("Only png, jpg, jpeg image is allowed"));
                        }
                    }
                },
            ]
        ];

        $validator = Validator::make($request->all(), $rules);
        if ($validator->fails()) {
            $errmsgs = $validator->getMessageBag()->add('error', 'true');
            return response()->json($validator->errors());
        }

        $user = User::where('id', $request->user_id)->first();
        if ($request->hasFile('preview_image')) {
            @unlink(public_path('assets/front/img/template-previews/' . $user->template_img));
            $filename = uniqid() . '.' . $prevImg->getClientOriginalExtension();
            $dir = public_path('assets/front/img/template-previews/');
            @mkdir($dir, 0775, true);
            $request->file('preview_image')->move($dir, $filename);
            $user->template_img = $filename;
        }
        $user->template_serial_number = $request->serial_number;
        $user->save();

        Session::flash('success', __('Updated Successfully'));
        return "success";
    }

    public function secretLogin(Request $request)
    {
        $user = User::where('id', $request->user_id)->first();
        if ($user) {
            if (Auth::guard('web')->login($user)) {
                return redirect()->route('user-dashboard')
                    ->withSuccess(__('You have Successfully loggedin'));
            }
        }
        return redirect("login")->withSuccess(__('Oppes! You have entered invalid credentials'));
    }
    public function tenant_lanuage_related_table_create($user)
    {
        $adminLangs = AdminLang::get();
        $langCount = User\Language::where('user_id', $user->id)->where('is_default', 1)->count();
        if ($langCount == 0) {
            foreach ($adminLangs as $lang) {
                //======language create==========
                $language = User\Language::create([
                    'name' => $lang->name,
                    'code' => $lang->code,
                    'is_default' => $lang->is_default,
                    'dashboard_default' => $lang->dashboard_default,
                    'rtl' => $lang->rtl,
                    'user_id' => $user->id,
                    'keywords' => $lang->customer_keywords
                ]);

                //====start menus=====
                $menus = '[{"text":"Home","href":"","icon":"empty","target":"_self","title":"","type":"home"},{"text":"Services","href":"","icon":"empty","target":"_self","title":"","type":"service"},{"text":"Rooms","href":"","icon":"empty","target":"_self","title":"","type":"room"},{"text":"Packages","href":"","icon":"empty","target":"_self","title":"","type":"package"},{"text":"Blog","href":"","icon":"empty","target":"_self","title":"","type":"blog"},{"text":"Gallery","href":"","icon":"empty","target":"_self","title":"","type":"gallery"},{"text":"Contact","href":"","icon":"empty","target":"_self","title":"","type":"contact"},{"type":"faq","text":"FAQ","href":"","target":"_self"}]';

                User\Menu::create([
                    'user_id' => $user->id,
                    'language_id' => $language->id,
                    'menus' => $menus,
                ]);

                //======= End Menu============

                // ======= Start Page Headings====
                PageHeading::create([
                    "language_id" => $language->id,
                    "user_id" => $user->id,
                    'gallery_page_title' => 'Gallery',
                    'rooms_page_title' => 'Rooms',
                    'reset_password_page_title' => 'Reset Password',
                    'packages_page_title' => 'Packages',
                    'dashboard_page_title' => 'Dashboard',
                    'services_page_title' => 'Services',
                    'room_bookings_page_title' => 'Room  Bookings',
                    "blog_page_title" => "Blog",
                    "package_bookings_page_title" => "Package Bookings",
                    "contact_page_title" => "Contact",
                    "error_page_title" => "404",
                    "faq_page_title" => "FAQ",
                    "forget_password_page_title" => "Forget Password",
                    "login_page_title" => "Login",
                    "signup_page_title" => "Sign Up",
                    "edit_profile_page_title" => "Edit Profile",
                    "change_password_page_title" => "Change Password",
                    "reset_password_page_title" => "Reset Password",
                    "created_at" => null,
                    "updated_at" => null
                ]);
                //End Page Heading

                //===Section Heading create
                SectionHeading::create([
                    "language_id" => $language->id,
                    "user_id" => $user->id,
                ]);
                //===End Section Heading
                User\BasicExtended::create([
                    'user_id' => $user->id,
                    'language_id' => $language->id,
                ]);
            }
        }
        return;
    }
}
